package com.centit.tablestore.service.impl;

import com.alibaba.fastjson2.JSONArray;
import com.centit.framework.common.ResponseData;
import com.centit.framework.security.model.CentitUserDetails;
import com.centit.support.algorithm.UuidOpt;
import com.centit.support.common.ObjectException;
import com.centit.support.database.ddl.GeneralDDLOperations;
import com.centit.support.database.metadata.SimpleTableField;
import com.centit.support.database.metadata.SimpleTableInfo;
import com.centit.support.database.utils.DBType;
import com.centit.support.database.utils.PageDesc;
import com.centit.support.database.utils.QueryUtils;
import com.centit.tablestore.dao.ProjectInfoDao;
import com.centit.tablestore.dao.ProjectTeamDao;
import com.centit.tablestore.dao.TableStructDao;
import com.centit.tablestore.po.ProjectInfo;
import com.centit.tablestore.po.TableStruct;
import com.centit.tablestore.service.TableStructService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

@Service("tableStructService")
@Transactional
public class TableStructServiceImpl implements TableStructService {

    @Autowired
    protected TableStructDao tableStructDao;

    @Autowired
    protected ProjectInfoDao projectInfoDao;

    @Autowired
    protected ProjectTeamDao projectTeamDao;

    @Override
    public void saveTableStruct(TableStruct tableStruct) {
        if(!projectTeamDao.checkMember(tableStruct.getProjectId(), tableStruct.getCreatorCode())){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+
                    tableStruct.getCreatorCode()+" 无权操作 " + tableStruct.getProjectId()+" 项目。");
        }

        if(tableStructDao.isTableExist(tableStruct.getProjectId(), tableStruct.getTableName())){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "表或者视图 "+
                    tableStruct.getTableName()+" 已存在。");
        }
        tableStructDao.saveNewObject(tableStruct);
    }

    @Override
    public void updateTableStruct(TableStruct tableStruct) {
        TableStruct dbStruct = tableStructDao.getTableByName(tableStruct.getProjectId(), tableStruct.getTableName());
        if(! StringUtils.equals(dbStruct.getTableId(), tableStruct.getTableId())){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "表或者视图 "+
                    tableStruct.getTableName()+" 已存在。");
        }
        tableStructDao.updateObject(tableStruct);
    }

    private List<SimpleTableField> findTableColumns(Map<String, List<SimpleTableField>> allTables, String tableName){
        if(StringUtils.isBlank(tableName)){
            if(allTables.size()==1){
                return allTables.values().iterator().next();
            } else {
                return null;
            }
        }
        //表名（别名）忽略大小写
        return allTables.get(tableName.toUpperCase());
    }

    private SimpleTableField searchTableColumn(Map<String, List<SimpleTableField>> allTables, String tableName, String columnName){
        List<SimpleTableField> columns = findTableColumns(allTables, tableName);
        if(columns != null){
            return findTableColumn(columns, columnName);
        }
        for(List<SimpleTableField> cols : allTables.values()){
            SimpleTableField column = findTableColumn(cols, columnName);
            if(column != null){
                return column;
            }
        }
        return null;
    }

    private SimpleTableField findTableColumn(List<SimpleTableField> columns, String columnName){
        for(SimpleTableField column : columns){
            //字段名忽略大小写
            if(StringUtils.equalsAnyIgnoreCase(columnName, column.getColumnName())){
                return column;
            }
        }
        return null;
    }

    /**
     * 解析试图的sql语句，并从对应的表中复制中文名称 lableName 和数据类型
     * @param tableStruct 试图对象
     * @return 解析后自动保存试图对象
     */
    @Override
    public TableStruct updateViewSql(TableStruct tableStruct) {
        String viewSql = tableStruct.extraViewSql();
        List<SimpleTableField> oldColumns = tableStruct.toTableInfo(DBType.MySql).getColumns();
        Pair<List<Pair<String, String>>, Map<String, String>> viewQueryStruct = QueryUtils.extraFieldAndTable(viewSql);

        //检查是否正确解析了 sql语句，如果没有直接返回
        if(viewQueryStruct == null || viewQueryStruct.getLeft()==null || viewQueryStruct.getRight()==null)
            return tableStruct;

        Map<String, List<SimpleTableField>> fromTables = new HashMap<>();
        for(Map.Entry<String, String> table : viewQueryStruct.getRight().entrySet()){
            TableStruct struct = tableStructDao.getTableByName(tableStruct.getProjectId(), table.getKey());
            if(struct!=null && struct.getMetadataJson()!=null){
                fromTables.put(table.getValue().toUpperCase(), struct.toTableInfo(DBType.MySql).getColumns());
            }
        }

        LinkedHashMap<String, SimpleTableField> newColumns = new LinkedHashMap<>();
        for (Pair<String, String> field : viewQueryStruct.getLeft()) {
            String fileDesc = field.getValue();
            String tableName, fieldFormula;
            int n = fileDesc.indexOf('.');
            if(n>0){
                tableName = fileDesc.substring(0,n).trim();
                fieldFormula = fileDesc.substring(n+1).trim();
            } else {
                tableName = "";
                fieldFormula = fileDesc;
            }
            if("*".equals(fieldFormula)){
                List<SimpleTableField> tableColumns = findTableColumns(fromTables, tableName);
                if(tableColumns!=null && tableColumns.size()>0) {
                    for (SimpleTableField field1 : tableColumns) {
                        SimpleTableField oldViewCol = findTableColumn(oldColumns, field1.getColumnName());
                        SimpleTableField column = new SimpleTableField();
                        if (oldViewCol != null) {
                            BeanUtils.copyProperties(oldViewCol, column);
                        } else {
                            BeanUtils.copyProperties(field1, column);
                        }
                        newColumns.put(field1.getColumnName(), column);
                    }
                }
            } else {
                SimpleTableField column = new SimpleTableField();
                SimpleTableField oldViewCol = findTableColumn(oldColumns, field.getKey());
                if(oldViewCol!=null){
                    BeanUtils.copyProperties(oldViewCol, column);
                } else {
                    SimpleTableField tempCol = searchTableColumn(fromTables , tableName, fieldFormula);
                    if(tempCol!=null){
                        BeanUtils.copyProperties(tempCol, column);
                    }
                }
                column.setColumnName(field.getKey());
                column.setColumnComment(fileDesc);
                newColumns.put(field.getKey(), column);
            }
        }

        tableStruct.setTableColumns(newColumns.values());
        //tableStructDao.saveNewObject(tableStruct);
        return tableStruct;
    }

    @Override
    public TableStruct updateFromSql(TableStruct tableStruct){
        String createSql = tableStruct.extraCreateSql();
        if(StringUtils.isBlank(createSql)){
            return tableStruct;
        }
        //tableStruct.clearCreateSql();
        SimpleTableInfo tableInfo = GeneralDDLOperations.parseDDL(createSql);
        if(StringUtils.isNotBlank(tableInfo.getTableName()))
            tableStruct.setTableName(tableInfo.getTableName());

        SimpleTableInfo table = tableStruct.toTableInfo(DBType.MySql);

        for(SimpleTableField column : tableInfo.getColumns()){
            SimpleTableField col = table.findFieldByColumn(column.getColumnName());
            if(col!=null){
                col.setFieldLabelName(column.getFieldLabelName());
                col.setColumnComment(column.getColumnComment());
                col.setColumnType(column.getColumnType());
                col.setMaxLength(column.getMaxLength());
                col.setScale(column.getScale());
                col.setMandatory(column.isMandatory());
                col.setPrimaryKey(column.isPrimaryKey());
                col.setDefaultValue(col.getDefaultValue());
            } else {
                table.addColumn(column);
            }
        }

        tableStruct.setTableColumns(table.getColumns());
        // tableStructDao.mergeObject(tableStruct);
        return tableStruct;
    }

    @Override
    public void deleteTableStruct(String tableId) {
        tableStructDao.deleteObjectById(tableId);
    }

    @Override
    public TableStruct getTableStruct(String tableId) {
        return tableStructDao.getObjectById(tableId);
    }

    @Override
    public TableStruct forkTable(CentitUserDetails userDetails, String tableId, String projectId) {
        if(!projectTeamDao.checkMember(projectId, userDetails.getUserCode())){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+ userDetails.getUserCode()+" 无权操作 " + projectId+" 项目。");
        }
        TableStruct ts = tableStructDao.getObjectById(tableId);
        if(ts == null){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "表 "+ tableId+" 不存在。");
        }
        //更新排序评分
        ProjectInfo projectInfo = projectInfoDao.getObjectById(ts.getProjectId());
        projectInfo.setTableForkedTimes(projectInfo.getTableForkedTimes()+1);
        projectInfoDao.updateSortScore(projectInfo, 0);

        //复制表
        ts.setTableId(UuidOpt.getUuidAsString22());
        ts.setCreatorCode(userDetails.getUserCode());
        ts.setCreatorName(userDetails.getUserInfo().getString("userName"));
        return tableStructDao.mergeTable(ts, projectId);
    }

    @Override
    public List<TableStruct> listProjectTables(String projectId, Map<String, Object> filterMap, PageDesc pageDesc) {
        filterMap.put("projectId", projectId);
        return tableStructDao.listObjectsByProperties(filterMap, pageDesc);
    }

    @Override
    public JSONArray statTables(String keyWord){
        return tableStructDao.statTables(keyWord);
    }

    @Override
    public JSONArray searchTables(String keyWord, String industryCategory, PageDesc pageDesc) {
        return tableStructDao.searchTables(keyWord, industryCategory, pageDesc);
    }

    @Override
    public String makeCreateSql(String tableId, DBType dbType) {
        TableStruct ts = tableStructDao.getObjectById(tableId);
        if(ts==null)
            return null;
        return ts.buildCreateTableSql(dbType);
    }

}
