package com.centit.tablestore.service.impl;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.centit.framework.common.ResponseData;
import com.centit.framework.jdbc.dao.DatabaseOptUtils;
import com.centit.framework.model.adapter.PlatformEnvironment;
import com.centit.framework.security.model.CentitUserDetails;
import com.centit.support.algorithm.*;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.DBType;
import com.centit.support.database.utils.PageDesc;
import com.centit.tablestore.dao.*;
import com.centit.tablestore.po.*;
import com.centit.tablestore.service.ProjectInfoService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service("projectInfoService")
@Transactional
public class ProjectInfoServiceImpl implements ProjectInfoService{

    @Autowired
    protected ProjectInfoDao projectInfoDao;

    @Autowired
    protected ProjectModuleDao projectModuleDao;

    @Autowired
    protected TableStructDao tableStructDao;

    @Autowired
    protected ProjectTeamDao projectTeamDao;

    @Autowired
    protected ProjectFollowerDao projectFollowerDao;

    @Autowired
    protected PlatformEnvironment platformEnvironment;

    private void saveProjectInfo(ProjectInfo projectInfo) {
        projectInfoDao.saveNewObject(projectInfo);
        ProjectTeam pt = new ProjectTeam(projectInfo.getProjectId(), projectInfo.getCreatorCode());
        pt.setMemberName(projectInfo.getCreatorName());
        projectTeamDao.saveNewObject(pt);
    }

    @Override
    public void createNewProject(ProjectInfo projectInfo) {
        saveProjectInfo(projectInfo);
        // 添加一个默认的模块，避免没有地方新建表
        ProjectModule projectModule = new ProjectModule();
        projectModule.setProjectId(projectInfo.getProjectId());
        projectModule.setCreatorCode(projectInfo.getCreatorCode());
        projectModule.setModuleName("默认模块");
        projectModule.setModuleDesc("随项目同时创建的默认模块。");
        projectModuleDao.saveNewObject(projectModule);
    }

    @Override
    public void updateProjectInfo(ProjectInfo projectInfo) {
        projectInfoDao.updateObject(projectInfo);
    }

    @Override
    public void deleteProjectInfo(String projectId) {
        projectInfoDao.deleteObjectById(projectId);
        projectModuleDao.deleteProjectModule(projectId);
        tableStructDao.deleteProjectTable(projectId);
        projectTeamDao.deleteProjectTeam(projectId);
    }

    @Override
    public ProjectInfo getProjectInfo(String projectId) {
        return projectInfoDao.getObjectById(projectId);
    }

    @Override
    public JSONArray listUserProjects(String userCode, boolean onlyPublic, PageDesc pageDesc) {
        String sqlSen = "select a.* from TS_PROJECT_INFO a join TS_PROJECT_TEAM b on(a.PROJECT_ID= b.PROJECT_ID) where b.PROJECT_MEMBER = ?";

        if(onlyPublic)
            sqlSen = sqlSen +" and a.VISIBILITY ='P'";
        return DatabaseOptUtils.listObjectsBySqlAsJson( projectInfoDao,
                sqlSen, new Object[]{userCode}, pageDesc);
    }

    @Override
    public JSONArray listFollowerProjects(String userCode, PageDesc pageDesc) {
        return DatabaseOptUtils.listObjectsBySqlAsJson( projectInfoDao,
                "select a.* from TS_PROJECT_INFO a join TS_PROJECT_FOLLOWER b on(a.PROJECT_ID= b.PROJECT_ID) where b.PROJECT_FOLLOWER = ?",
                new Object[]{userCode},
                pageDesc);
    }

    @Override
    public Map<String, Object> getUserStatInfo(String userCode){
         int projects = NumberBaseOpt.castObjectToInteger(
                DatabaseOptUtils.getScalarObjectQuery(projectTeamDao,
                        "select count(1) as projectSum from TS_PROJECT_TEAM a " +
                                "where a.PROJECT_MEMBER = ?",
                        new Object[]{ userCode}), 0);

        int tables = NumberBaseOpt.castObjectToInteger(
                DatabaseOptUtils.getScalarObjectQuery(projectTeamDao,
                        "select count(1) as tableSum from TS_TABLE_STRUCT a " +
                                "where a.CREATOR_CODE = ?",
                        new Object[]{ userCode}), 0);
        int moudles = NumberBaseOpt.castObjectToInteger(
                DatabaseOptUtils.getScalarObjectQuery(projectTeamDao,
                        "select count(1) as moduleSum from TS_PROJECT_MODULE a " +
                                "where a.CREATOR_CODE = ?",
                        new Object[]{ userCode}), 0);

        Map<String, Object> objectMap = CollectionsOpt.createHashMap("projectSum", projects,
                "tableSum", tables, "moduleSum", moudles, "userCode", userCode);

        CentitUserDetails ud = platformEnvironment.loadUserDetailsByUserCode(userCode);
        if(ud != null){
            objectMap.put("userName", ud.getUserInfo().getString("userName"));
            objectMap.put("email", ud.getUserInfo().getString("regEmail"));
        }
        return objectMap;
    }

    @Override
    public JSONArray listUserCreateProjects(String userCode, PageDesc pageDesc) {
        return DatabaseOptUtils.listObjectsBySqlAsJson( projectInfoDao,
               "select a.* from TS_PROJECT_INFO a where a.CREATOR_CODE=?",
                new Object[]{userCode}, pageDesc);
    }

    @Override
    public void addProjectMember(String optUser, String projectId, String memberCode) {
        if(!projectInfoDao.checkOwner(projectId, optUser)){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+optUser+" 不是项目 " + projectId+" 的创建人，无权操作。");
        }

        ProjectTeam pt = projectTeamDao.getObjectById(new ProjectTeamId(projectId, memberCode));
        if(pt!=null)
            return;
        //CodeRepositoryUtil.getUserInfoByCode()
        pt = new ProjectTeam(projectId, memberCode);
        CentitUserDetails ud = platformEnvironment.loadUserDetailsByUserCode(memberCode);
        if(ud != null){
            pt.setMemberName(ud.getUserInfo().getString("userName"));
        }
        projectTeamDao.saveNewObject(pt);
    }

    @Override
    public void deleteProjectMember(String optUser, String projectId, String memberCode) {
        if(!projectInfoDao.checkOwner(projectId, optUser)){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+optUser+" 不是项目 " + projectId+" 的创建人，无权操作。");
        }
        if(!optUser.equals(memberCode)){
            projectTeamDao.deleteObjectById(new ProjectTeamId(projectId, memberCode));
        }
    }

    @Override
    public List<ProjectTeam> listProjectMember(String optUser, String projectId) {
        if(!projectInfoDao.checkOwner(projectId, optUser)){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+optUser+" 不是项目 " + projectId+" 的创建人，无权操作。");
        }
        return projectTeamDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
    }

    @Override
    public void addFollowerProject(ProjectFollower follower){
        ProjectFollower pf = projectFollowerDao.getObjectById(follower.getPfId());
        if(pf==null) {
            projectFollowerDao.saveNewObject(follower);
            ProjectInfo projectInfo = projectInfoDao.getObjectById(follower.getProjectId());
            projectInfo.setFollowedSum(projectInfo.getFollowedSum() + 1);
            projectInfoDao.updateSortScore(projectInfo, 1);
        }
    }

    @Override
    public void deleteFollowerProject(String optUser, String projectId){
        ProjectFollowerId fid = new ProjectFollowerId(projectId, optUser);
        ProjectFollower pf = projectFollowerDao.getObjectById(fid);
        if(pf!=null) {
            projectFollowerDao.deleteObjectById(fid);
            ProjectInfo projectInfo = projectInfoDao.getObjectById(projectId);
            projectInfo.setFollowedSum(projectInfo.getFollowedSum() - 1);
            projectInfoDao.updateSortScore(projectInfo, 2);
        }
    }

    @Override
    public ProjectInfo forkProject(CentitUserDetails userDetails, String projectId) {
        ProjectInfo projectInfo = projectInfoDao.getObjectById(projectId);
        if(projectInfo==null) {
            throw new ObjectException(ResponseData.ERROR_OPERATION, "项目 " + projectId+" 不存在。");
        }
        if(StringUtils.equals(userDetails.getUserCode(), projectInfo.getCreatorCode())) {
            throw new ObjectException(ResponseData.ERROR_OPERATION, "用户 " + userDetails.getUserCode() +" 不能fork自己的项目。");
        }

        ProjectInfo forkProject = projectInfoDao.getObjectByProperties(
                CollectionsOpt.createHashMap("creatorCode", userDetails.getUserCode(), "forkProject", projectId));

        if(forkProject!=null) {
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "已经存在一个相同的fork项目，避免数据冗余不能再次fork，" +
                    "可以用导入导出来更新最新的项目数据。");
        }

        String userName = userDetails.getUserInfo().getString("userName");
        //更新排序评分
        projectInfo.setProjectForkedTimes(projectInfo.getProjectForkedTimes()+1);
        projectInfoDao.updateSortScore(projectInfo, 0);
        //复制项目
        forkProject = projectInfo;
        forkProject.setProjectId(UuidOpt.getUuidAsString22());
        forkProject.setForkProject(projectId);
        forkProject.setCreatorCode(userDetails.getUserCode());
        forkProject.setCreatorName(userName);
        forkProject.clearSortScore();

        //colone 主表
        saveProjectInfo(forkProject);
        //colone 字表 TS_PROJECT_MODULE
        HashMap<String, String> tableIdMap = new HashMap<>(100);
        //colone 字表 TS_TABLE_STRUCT
        List<TableStruct> tables = tableStructDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
        if(tables!=null){
            for(TableStruct ts : tables){
                String newTableId = UuidOpt.getUuidAsString22();
                tableIdMap.put(ts.getTableId(), newTableId);
                ts.setTableId(newTableId);
                ts.setProjectId(forkProject.getProjectId());
                ts.setCreatorCode(userDetails.getUserCode());
                ts.setCreatorName(userName);
                tableStructDao.saveNewObject(ts);
            }
        }

        List<ProjectModule> modules = projectModuleDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
        if(modules!=null){
            for(ProjectModule pm : modules){
                pm.setModuleId(UuidOpt.getUuidAsString22());
                pm.setProjectId(forkProject.getProjectId());
                pm.setCreatorCode(userDetails.getUserCode());

                JSONObject moduleDesign = pm.getModuleDesign();
                if(moduleDesign!=null) {
                    Object moduleTables = moduleDesign.get("tables");
                    if (moduleTables instanceof List) {
                        for (Object table : (List<Object>) moduleTables) {
                            if (table instanceof Map) {
                                Map<String, Object> tableMap = (Map<String, Object>) table;
                                Object tableInfo = tableMap.get("info");
                                if (tableInfo instanceof JSONObject) {
                                    String oldTalbeId =
                                            StringBaseOpt.castObjectToString(((JSONObject) tableInfo).get("tableId"));
                                    String newTableId = tableIdMap.get(oldTalbeId);
                                    if (newTableId == null) {
                                        newTableId = "";
                                    }
                                    ((JSONObject) tableInfo).put("tableId", newTableId);
                                }
                            }
                        }
                    }

                    Object views = moduleDesign.get("views");
                    if (views instanceof List) {
                        for (Object view : (List<Object>) views) {
                            if (view instanceof Map) {
                                Map<String, Object> viewMap = (Map<String, Object>) view;
                                Object viewInfo = viewMap.get("info");
                                if (viewInfo instanceof JSONObject) {
                                    String oldTalbeId =
                                            StringBaseOpt.castObjectToString(((JSONObject) viewInfo).get("viewId"));
                                    String newTableId = tableIdMap.get(oldTalbeId);
                                    if (newTableId == null) {
                                        newTableId = "";
                                    }
                                    ((JSONObject) viewInfo).put("viewId", newTableId);
                                }
                            }
                        }
                    }
                }
                projectModuleDao.saveNewObject(pm);
            }
        }

        return forkProject;
    }

    @Override
    public boolean hasFollowedProject(String projectId, String userCode){
        return projectFollowerDao.hasFollowed(projectId, userCode);
    }

    @Override
    public boolean checkMember(String projectId, String userCode){
        return projectTeamDao.checkMember(projectId, userCode);
    }
    @Override
    public void mergeProjectTables(String projectId, CentitUserDetails userDetails, List<TableStruct> tableStructs){
        if(!projectTeamDao.checkMember(projectId, userDetails.getUserCode())){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+ userDetails.getUserCode()+" 无权操作 " + projectId+" 项目。");
        }
        String userName = userDetails.getUserInfo().getString("userName");
        for(TableStruct struct : tableStructs){
            struct.setProjectId(projectId);
            struct.setCreatorCode(userDetails.getUserCode());
            struct.setCreatorName(userName);
            tableStructDao.mergeTable(struct, projectId);
        }
    }

    @Override
    public String makeCreateSql(String projectId, DBType dbType) {
        StringBuilder sqlBuilder = new StringBuilder("/*==========================================*/\r\n" +
                "/* Created on:   ").append(DatetimeOpt.currentDatetime()).append("                        */\r\n" +
                "/*==========================================*/\r\n");
        List<TableStruct> tables = tableStructDao.listProjectTableWithColumn(projectId);
        StringBuilder viewSqlBuilder = new StringBuilder();

        if(tables!=null){
            for(TableStruct struct : tables){
                if("V".equals(struct.getTableType())) {
                    viewSqlBuilder.append("/*===  ").append(struct.getTableLabelName()).append("  =======*/\r\n")
                            .append(struct.buildCreateTableSql(dbType)).append("\r\n\r\n");
                } else {
                    sqlBuilder.append("/*===  ").append(struct.getTableLabelName()).append("  =======*/\r\n")
                            .append(struct.buildCreateTableSql(dbType)).append("\r\n\r\n");
                }
            }
        }
        sqlBuilder.append(viewSqlBuilder);
        return sqlBuilder.toString();
    }

    @Override
    public JSONObject exportProject(String projectId) {
        JSONObject project = new JSONObject();
        project.put("projectInfo", projectInfoDao.getObjectById(projectId));

        List<ProjectModule> modules = projectModuleDao.listProjectModuleWithDesign(projectId);
        if(modules!=null && modules.size()>0) {
            JSONArray jaModules = new JSONArray();
            for (ProjectModule module : modules) {
                jaModules.add(module.toModuleJson());
            }
            project.put("modules", jaModules);
        }

        List<TableStruct> tables = tableStructDao.listProjectTableWithColumn(projectId);
        if(tables!=null && tables.size()>0) {
            JSONArray jaTables = new JSONArray();
            for (TableStruct table : tables) {
                jaTables.add(table.toTableJson());
            }
            project.put("tables", jaTables);
        }
        return project;
    }

}