package com.centit.tablestore.service.impl;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.centit.framework.common.ResponseData;
import com.centit.support.algorithm.UuidOpt;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.PageDesc;
import com.centit.tablestore.dao.*;
import com.centit.tablestore.po.ProjectHistory;
import com.centit.tablestore.po.ProjectInfo;
import com.centit.tablestore.po.ProjectModule;
import com.centit.tablestore.po.TableStruct;
import com.centit.tablestore.service.ProjectHistoryService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service("projectHistoryService")
@Transactional
public class ProjectHistoryServiceImpl implements ProjectHistoryService {

    @Autowired
    protected ProjectModuleDao projectModuleDao;

    @Autowired
    protected ProjectInfoDao projectInfoDao;

    @Autowired
    protected TableStructDao tableStructDao;

    @Autowired
    protected ProjectHistoryDao projectHistoryDao;

    @Autowired
    protected ProjectTeamDao projectTeamDao;

    @Override
    public void saveProjectHistory(ProjectHistory historyInfo) {
        projectHistoryDao.saveNewObject(historyInfo);
    }

    @Override
    public void updateProjectHistory(ProjectHistory historyInfo) {
        projectHistoryDao.updateObject(historyInfo);
    }

    @Override
    public void deleteProjectHistory(String historyId) {
        projectHistoryDao.deleteProjectHistory(historyId);
    }

    @Override
    public ProjectHistory getProjectHistory(String historyId) {
        return projectHistoryDao.getObjectById(historyId);
    }

    @Override
    public void restoreProjectHistory(String historyId, String userCode) {
        ProjectHistory history = projectHistoryDao.getObjectById(historyId);
        if(history==null){
            throw new ObjectException(ObjectException.DATA_NOT_FOUND_EXCEPTION, "没有找到历史版本信息："+historyId+" 。");
        }
        restoreProject(history.getProjectId(), history.getStructContent(), userCode);
    }

    @Override
    public List<ProjectHistory> listProjectHistory(Map<String, Object> params, PageDesc pageDesc){
        return projectHistoryDao.listObjectsByProperties(params, pageDesc);
    }

    public void restoreOrMergeProject(String projectId, JSONObject projectInfo, String userCode, boolean merge) {

        if(!projectTeamDao.checkMember(projectId, userCode)){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+ userCode+" 无权操作 " + projectId+" 项目。");
        }

        //删除项目对应的表 和 模块
        JSONObject project = projectInfo.getJSONObject("projectInfo");
        ProjectInfo pi = projectInfoDao.getObjectById(projectId);
        if(pi==null){
            throw new ObjectException(ObjectException.DATA_NOT_FOUND_EXCEPTION, "没有找到响应的项目信息："+projectId+" 。");
        }

        pi.setProjectName(project.getString("projectName"));
        pi.setIndustryCategory(project.getString("industryCategory"));
        pi.setProjectDesc(project.getString("projectDesc"));
        projectInfoDao.updateObject(pi);

        //替换表； 由于ID的原因，简单粗暴的全部删除再插入
        if(!merge) {
            tableStructDao.deleteProjectTable(projectId);
        }
        Map<String, String> tableIdMap = new HashMap<>(128);
        JSONArray tables = projectInfo.getJSONArray("tables");
        if(tables!=null && tables.size()>0) {
            for(Object obj : tables){
                if(obj instanceof JSONObject) {
                    TableStruct ts = TableStruct.formTableJson((JSONObject) obj);
                    if(StringUtils.isNotBlank(ts.getTableName())){
                        String newTableId = UuidOpt.getUuidAsString22();
                        tableIdMap.put(ts.getTableId(), newTableId);
                        ts.setTableId(newTableId);
                        ts.setProjectId(projectId);
                        //保存新的对象
                        if(merge) {
                            tableStructDao.mergeTable(ts, projectId);
                        } else {
                            tableStructDao.saveNewObject(ts);
                        }
                    }
                }
            }
        }

        //替换模块
        if(!merge) {
            projectModuleDao.deleteProjectModule(projectId);
        }
        JSONArray modules = projectInfo.getJSONArray("modules");
        if(modules!=null && modules.size()>0) {
            for(Object obj : modules){
                if(obj instanceof JSONObject) {
                    ProjectModule pm = ProjectModule.formModuleJson((JSONObject) obj);
                    if(StringUtils.isNotBlank(pm.getModuleName())){
                        pm.setProjectId(projectId);
                        pm.setModuleId(UuidOpt.getUuidAsString22());
                        //替换 表和视图中的对应关系
                        pm.replaceModuleTablesId(tableIdMap);
                        //保存新的对象
                        if(merge) {
                            projectModuleDao.mergeProjectModule(pm);
                        } else {
                            projectModuleDao.saveNewObject(pm);
                        }
                    }
                }
            }
        }

    }

    @Override
    public void restoreProject(String projectId, JSONObject projectInfo, String userCode){
        restoreOrMergeProject(projectId, projectInfo, userCode, false);
    }

    @Override
    public void mergeProject(String projectId, JSONObject projectInfo, String userCode){
        restoreOrMergeProject(projectId, projectInfo, userCode,true);
    }
}
