package com.centit.tablestore.controller;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.centit.fileserver.utils.SystemTempFileUtils;
import com.centit.fileserver.utils.UploadDownloadUtils;
import com.centit.framework.common.JsonResultUtils;
import com.centit.framework.common.ResponseData;
import com.centit.framework.common.WebOptUtils;
import com.centit.framework.core.controller.BaseController;
import com.centit.framework.core.controller.WrapUpContentType;
import com.centit.framework.core.controller.WrapUpResponseBody;
import com.centit.framework.core.dao.PageQueryResult;
import com.centit.framework.security.model.CentitUserDetails;
import com.centit.support.algorithm.BooleanBaseOpt;
import com.centit.support.algorithm.StringBaseOpt;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.DBType;
import com.centit.support.database.utils.PageDesc;
import com.centit.support.file.FileIOOpt;
import com.centit.support.file.FileSystemOpt;
import com.centit.tablestore.po.ProjectFollower;
import com.centit.tablestore.po.ProjectInfo;
import com.centit.tablestore.po.ProjectTeam;
import com.centit.tablestore.po.TableStruct;
import com.centit.tablestore.service.ProjectHistoryService;
import com.centit.tablestore.service.ProjectInfoService;
import com.centit.tablestore.service.impl.PdmTableInfoUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Api(value = "项目维护接口", tags = "项目维护接口")
@RestController
@RequestMapping(value = "project")
public class ProjectController extends BaseController {

    @Autowired
    protected ProjectInfoService projectInfoService;

    @Autowired
    private ProjectHistoryService projectHistoryService;

    @ApiOperation(value = "获取用户所有项目", notes = "获取用户所有项目")
    @ApiImplicitParam(
        name = "userCode", value = "用户代码，缺省获取当前用户代码",
        paramType = "query")
    @GetMapping("/stat")
    @WrapUpResponseBody
    public Map<String, Object> calcUserStatInfo(HttpServletRequest request) {
        String userCode = request.getParameter("userCode");
        //Map<String, Object> filterMap = collectRequestParameters(request);
        if(StringUtils.isBlank(userCode)){
            userCode = WebOptUtils.getCurrentUserCode(request);
        }
        return projectInfoService.getUserStatInfo(userCode);
    }

    @ApiOperation(value = "获取用户所有项目", notes = "获取用户所有项目")
    @ApiImplicitParams({
    @ApiImplicitParam(
            name = "pageDesc", value = "json格式，分页对象信息",
            paramType = "body", dataTypeClass = PageDesc.class),
    @ApiImplicitParam(
            name = "onlyCreated", value = "仅仅返回用户创建的项目",
            paramType = "query"),
    @ApiImplicitParam(
            name = "userCode", value = "用户代码，缺省获取当前用户代码",
            paramType = "query")})
    @GetMapping
    @WrapUpResponseBody
    public PageQueryResult<Object> list(PageDesc pageDesc, String onlyCreated, HttpServletRequest request) {
        boolean onlyPublic;
        String userCode = request.getParameter("userCode");
        String currentUser = WebOptUtils.getCurrentUserCode(request);
        if(StringUtils.isBlank(userCode)){
            userCode = currentUser;
            onlyPublic = false;
        } else {
            onlyPublic = ! StringUtils.equals(userCode, currentUser);
        }
        if(StringUtils.isBlank(userCode)){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "请指定要查询的用户代码。");
        }

        JSONArray projectInfos;
        if(BooleanBaseOpt.castObjectToBoolean(onlyCreated, false)) {
            projectInfos = projectInfoService.listUserCreateProjects(userCode, pageDesc);
        } else {
            projectInfos = projectInfoService.listUserProjects(userCode, onlyPublic, pageDesc);
        }
        return PageQueryResult.createJSONArrayResult(projectInfos, pageDesc, ProjectInfo.class);
    }


    @ApiOperation(value = "获取用户关注的项目", notes = "获取用户关注的项目")
    @ApiImplicitParam(
            name = "pageDesc", value = "json格式，分页对象信息",
            paramType = "body", dataTypeClass = PageDesc.class)
    @GetMapping("/followed")
    @WrapUpResponseBody
    public PageQueryResult<Object> followedProjects(PageDesc pageDesc,  HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        if(StringUtils.isBlank(userCode)){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "获取用户信息失败，只有登录的用户可以获取到他关注的项目。");
        }
        JSONArray projectInfos = projectInfoService.listFollowerProjects(userCode, pageDesc);
        return PageQueryResult.createJSONArrayResult(projectInfos, pageDesc, ProjectInfo.class);
    }

    @ApiOperation(value = "添加关注项目")
    @PostMapping(value = "/followed")
    @WrapUpResponseBody
    public void addFollowedProject(@RequestBody ProjectFollower follower, HttpServletRequest request) {
        //projectId projectMember
        CentitUserDetails userDetails = WebOptUtils.getCurrentUserDetails(request);
        if(userDetails == null){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "获取用户信息失败，只有登录的用户可以添加关注的项目。");
        }

        if(StringUtils.isBlank(follower.getProjectId())){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "请输入正确的项目代码ProjectId。");
        }
        follower.setProjectFollower(userDetails.getUserCode());
        follower.setFollowerName(userDetails.getUserInfo().getString("userName"));
        projectInfoService.addFollowerProject(follower);
    }

    @ApiOperation(value = "删除关注项目")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @DeleteMapping(value = "/followed/{projectId}")
    @WrapUpResponseBody
    public void deleteFollowedProject(@PathVariable String projectId, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        if(StringUtils.isBlank(userCode)){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "获取用户信息失败，只有登录的用户可以取消他的关注项目。");
        }
        projectInfoService.deleteFollowerProject(userCode, projectId);
    }

    @ApiOperation(value = "是否已经关注项目")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping(value = "/followed/{projectId}")
    @WrapUpResponseBody
    public boolean hasFollowedProject(@PathVariable String projectId, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        if(StringUtils.isBlank(userCode)){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "获取用户信息失败，只有登录的用户可以检测是否关注项目。");
        }
        return projectInfoService.hasFollowedProject(projectId, userCode);
    }

    @ApiOperation(value = "查询单个项目")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping(value = "/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public ProjectInfo getProjectInfo(@PathVariable String projectId) {
        return projectInfoService.getProjectInfo(projectId);
    }

    @ApiOperation(value = "删除单个项目")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @DeleteMapping(value = "/{projectId}")
    @WrapUpResponseBody
    public void deleteProjectInfo(@PathVariable String projectId) {
        projectInfoService.deleteProjectInfo(projectId);
    }

    @ApiOperation(value = "新建项目数据")
    @PostMapping
    @WrapUpResponseBody
    public ProjectInfo saveProjectInfo(@RequestBody ProjectInfo projectInfo, HttpServletRequest request){
        CentitUserDetails ud = WebOptUtils.getCurrentUserDetails(request);
        if(ud == null){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "Session过期或者用户没有登录。");
        }
        projectInfo.setCreatorCode(ud.getUserCode());
        projectInfo.setCreatorName(ud.getUserInfo().getString("userName"));
        projectInfoService.createNewProject(projectInfo);
        return projectInfo;
    }

    @ApiOperation(value = "修改项目数据")
    @PutMapping
    @WrapUpResponseBody
    public void updateProjectInfo(@RequestBody ProjectInfo projectInfo){
        projectInfoService.updateProjectInfo(projectInfo);
    }

    @ApiOperation(value = "生成脚本", notes = "生成脚本")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID"),
        @ApiImplicitParam(name = "dbtype", type = "query", value = "数据类别")})
    @GetMapping("/sql/{projectId}")
    @WrapUpResponseBody
    public String createSql(@PathVariable String projectId, String dbtype) {
        return projectInfoService.makeCreateSql(projectId, DBType.mapDBType(dbtype, DBType.MySql));
    }

    @ApiOperation(value = "DDL脚本以文件方式返回", notes = "DDL脚本以文件方式返回")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID"),
            @ApiImplicitParam(name = "dbtype", type = "query", value = "数据类别")})
    @GetMapping("/ddl/{projectId}")
    public void downloadDDL(@PathVariable String projectId, String dbtype, HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        ProjectInfo projectInfo = projectInfoService.getProjectInfo(projectId);
        if(projectInfo==null){
            throw new ObjectException(ObjectException.DATA_NOT_FOUND_EXCEPTION ,"项目："+projectId+" 不存在！");
        }
        String ddl = projectInfoService.makeCreateSql(projectId, DBType.mapDBType(dbtype, DBType.MySql));
        String fileName = projectInfo.getProjectName()+".sql";
        ByteArrayInputStream bis = new ByteArrayInputStream(ddl.getBytes(StandardCharsets.UTF_8));
        UploadDownloadUtils.downloadFile(bis, fileName, request, response);
    }

    @ApiOperation(value = "导出项目的表结构信息", notes = "导出项目的表结构信息")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping("/export/{projectId}")
    public void exportProject(@PathVariable String projectId, HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        ProjectInfo projectInfo = projectInfoService.getProjectInfo(projectId);
        if(projectInfo==null){
            throw new ObjectException(ObjectException.DATA_NOT_FOUND_EXCEPTION ,"项目："+projectId+" 不存在！");
        }
        JSONObject json = projectInfoService.exportProject(projectId);
        String fileName = projectInfo.getProjectName()+".json";
        ByteArrayInputStream bis = new ByteArrayInputStream(json.toJSONString().getBytes(StandardCharsets.UTF_8));
        UploadDownloadUtils.downloadFile(bis, fileName, request, response);
    }

    @ApiOperation(value = "fork项目数据")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @PostMapping("/fork/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public ProjectInfo forkProjectInfo(@PathVariable String projectId, HttpServletRequest request){
        CentitUserDetails ud = WebOptUtils.getCurrentUserDetails(request);
        if(ud == null){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "Session过期或者用户没有登录。");
        }
        return projectInfoService.forkProject(ud, projectId);
    }

    @ApiOperation(value = "查询项目协作人员")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping(value = "/member/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public List<ProjectTeam>  listProjectMember(@PathVariable String projectId, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        return projectInfoService.listProjectMember(userCode, projectId);
    }

    @ApiOperation(value = "检查自己是否可以编辑项目")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping(value = "/canedit/{projectId}")
    @WrapUpResponseBody
    public boolean canEditProject(@PathVariable String projectId, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        return projectInfoService.checkMember(projectId, userCode);
    }

    @ApiOperation(value = "添加项目协作人员")
    @PostMapping(value = "/member")
    @WrapUpResponseBody
    public void addProjectMember(@RequestBody String projectMembers, HttpServletRequest request) {
        //projectId projectMember
        String userCode = WebOptUtils.getCurrentUserCode(request);
        JSONObject projMembers = JSONObject.parseObject(projectMembers);
        String projectId = projMembers.getString("projectId");
        Object members = projMembers.get("projectMember");
        if(members == null )
            members = projMembers.get("projectMembers");
        if(StringUtils.isBlank(userCode) || StringUtils.isBlank(projectId) || members==null){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "表单数据格式不正确，或者用户没有登录。");
        }
        List<String> memeberCodes = StringBaseOpt.objectToStringList(members);
        for(String mc : memeberCodes) {
            projectInfoService.addProjectMember(userCode, projectId, mc);
        }
    }

    @ApiOperation(value = "删除项目协作人员")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID"),
        @ApiImplicitParam(name = "memberCode", type = "path", value = "协作人员代码")})
    @DeleteMapping(value = "/delete/{projectId}/{memberCode}")
    @WrapUpResponseBody
    public void deleteProjectMember(@PathVariable String projectId, @PathVariable String memberCode, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        projectInfoService.deleteProjectMember(userCode, projectId, memberCode);
    }

    @ApiOperation(value = "删除项目协作人员,post方法，可以批量删除")
    @PostMapping(value = "/deleteMembers")
    @WrapUpResponseBody
    public void deleteProjectMember(@RequestBody String projectMembers, HttpServletRequest request) {
        //projectId projectMember
        String userCode = WebOptUtils.getCurrentUserCode(request);
        JSONObject projMembers = JSONObject.parseObject(projectMembers);
        String projectId = projMembers.getString("projectId");
        Object members = projMembers.get("projectMember");
        if(members == null )
            members = projMembers.get("projectMembers");
        if(StringUtils.isBlank(userCode) || StringUtils.isBlank(projectId) || members==null){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "表单数据格式不正确，或者用户没有登录。");
        }
        List<String> memeberCodes = StringBaseOpt.objectToStringList(members);
        for(String mc : memeberCodes) {
            projectInfoService.deleteProjectMember(userCode, projectId,mc);
        }
    }

    @ApiOperation(value = "检查文件完整性")
    @CrossOrigin(origins = "*", allowCredentials = "true", maxAge = 86400, methods = RequestMethod.GET)
    @RequestMapping(value = "/range", method = {RequestMethod.GET})
    @WrapUpResponseBody
    public JSONObject checkFileRange(String token, long size) {
        //FileRangeInfo fr = new FileRangeInfo(token,size);
        //Pair<String, InputStream> fileInfo = UploadDownloadUtils.fetchInputStreamFromMultipartResolver(request);
        //检查临时目录中的文件大小，返回文件的其实点
        String tempFilePath = SystemTempFileUtils.getTempFilePath(token, size);
        long tempFileSize = SystemTempFileUtils.checkTempFileSize(tempFilePath);
        Map<String, Object> data = new HashMap<>(4);
        data.put("tempFilePath", token + "_" + size);
        JSONObject jsonObject = UploadDownloadUtils.makeRangeUploadJson(tempFileSize, token, token + "_" + size);
        if (tempFileSize == size) {
            data.put("tables", PdmTableInfoUtils.importTableFromPdm(tempFilePath));
            jsonObject.put("tables", data);
        }
        return jsonObject;
    }

    @ApiOperation(value = "导入pdm返回表数据")
    @CrossOrigin(origins = "*", allowCredentials = "true", maxAge = 86400, methods = RequestMethod.POST)
    @RequestMapping(value = "/range", method = {RequestMethod.POST})
    public void syncPdm(String token, long size,
                        HttpServletRequest request, HttpServletResponse response)
            throws IOException {

        Pair<String, InputStream> fileInfo = UploadDownloadUtils.fetchInputStreamFromMultipartResolver(request);
        FileSystemOpt.createDirect(SystemTempFileUtils.getTempDirectory());
        String tempFilePath = SystemTempFileUtils.getTempFilePath(token, size);
        try (FileOutputStream out = new FileOutputStream(new File(tempFilePath))) {
            long uploadSize = FileIOOpt.writeInputStreamToOutputStream(fileInfo.getRight(), out);
            if (uploadSize > 0) {
                //上传到临时区成功
                JSONObject jsonObject = new JSONObject();
                Map<String, Object> data = new HashMap<>(4);
                data.put("tempFilePath", token + "_" + size);
                data.put("tables", PdmTableInfoUtils.getTableNameFromPdm(tempFilePath));
                jsonObject.put("tables", data);
                JsonResultUtils.writeSingleDataJson(jsonObject, response);
            } else {
                JsonResultUtils.writeOriginalJson(UploadDownloadUtils.
                        makeRangeUploadJson(uploadSize, token, token + "_" + size).toJSONString(), response);
            }

        } catch (ObjectException e) {
            logger.error(e.getMessage(), e);
            JsonResultUtils.writeHttpErrorMessage(e.getExceptionCode(),
                    e.getMessage(), response);
        }
    }


    @ApiOperation(value = "确认导入pdm修改表元数据表")
    @RequestMapping(value = "/confirm/{projectId}", method = {RequestMethod.POST})
    @WrapUpResponseBody
    public void importPdmConfirm(@PathVariable String projectId, @RequestBody String data,
                            HttpServletRequest request) {
        CentitUserDetails ud = WebOptUtils.getCurrentUserDetails(request);
        if(ud == null){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "Session过期或者用户没有登录。");
        }

        Map<String, Object> params = collectRequestParameters(request);
        JSONObject object = JSON.parseObject(data);
        object.putAll(params);
        String tempFilePath = SystemTempFileUtils.getTempDirectory() + object.getString("tempFilePath") + ".tmp";
        JSONArray jsonArray = object.getJSONArray("data");
        List<String> tables = new ArrayList<>();
        for (Object o : jsonArray) {
            tables.add(o.toString());
        }
        // save tables
        List<TableStruct> tableStructs = PdmTableInfoUtils.importTableFromPdm(tempFilePath, tables);
        projectInfoService.mergeProjectTables(projectId, ud, tableStructs);
    }

    @ApiOperation(value = "导入TableStore中导入表结构")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @RequestMapping(value = "/imoprt/{projectId}", method = RequestMethod.POST)
    @WrapUpResponseBody
    public void importFromTableStore(@PathVariable String projectId, HttpServletRequest request) throws IOException{
        String userCode = WebOptUtils.getCurrentUserCode(request);
        if(StringUtils.isBlank(userCode)){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "Session过期或者用户没有登录。");
        }
        Pair<String, InputStream> fileInfo = UploadDownloadUtils.fetchInputStreamFromMultipartResolver(request);
        JSONObject jsonObject = JSON.parseObject(fileInfo.getRight());
        if(jsonObject!=null && //检验json的合法性
            jsonObject.containsKey("projectInfo") && jsonObject.containsKey("tables") && jsonObject.containsKey("modules")){

            projectHistoryService.mergeProject(projectId, jsonObject, userCode);
        } else {
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "文件中的json格式不正确！");
        }
    }
}
