package com.centit.tablestore.service.impl;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.centit.framework.common.ResponseData;
import com.centit.framework.security.model.CentitUserDetails;
import com.centit.support.algorithm.CollectionsOpt;
import com.centit.support.algorithm.DatetimeOpt;
import com.centit.support.algorithm.StringBaseOpt;
import com.centit.support.algorithm.UuidOpt;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.DBType;
import com.centit.tablestore.dao.ProjectInfoDao;
import com.centit.tablestore.dao.ProjectModuleDao;
import com.centit.tablestore.dao.ProjectTeamDao;
import com.centit.tablestore.dao.TableStructDao;
import com.centit.tablestore.po.ProjectInfo;
import com.centit.tablestore.po.ProjectModule;
import com.centit.tablestore.po.TableStruct;
import com.centit.tablestore.service.ProjectModuleService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Service("projectModuleService")
@Transactional
public class ProjectModuleServiceImpl implements ProjectModuleService {

    @Autowired
    protected ProjectModuleDao projectModuleDao;

    @Autowired
    protected ProjectInfoDao projectInfoDao;

    @Autowired
    protected TableStructDao tableStructDao;

    @Autowired
    protected ProjectTeamDao projectTeamDao;

    @Override
    public void saveProjectModule(ProjectModule projectModule) {
        projectModuleDao.saveNewObject(projectModule);
    }

    @Override
    public void updateProjectModule(ProjectModule projectModule) {
        projectModuleDao.updateObject(projectModule);
    }

    @Override
    public void deleteProjectModule(String moduleId) {
        projectModuleDao.deleteObjectById(moduleId);
    }

    @Override
    public ProjectModule getProjectModule(String moduleId) {
        return projectModuleDao.getObjectById(moduleId);
    }

    @Override
    public List<ProjectModule> listProjectModules(String projectId) {
        return projectModuleDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
    }

    @Override
    public List<TableStruct> listModuleTables(String moduleId) {
        ProjectModule pm = projectModuleDao.getObjectById(moduleId);
        if (pm == null) return null;
        Object tables = pm.getModuleDesign().get("tables");
        List<String> tableIds = new ArrayList<>(64);
        if (tables instanceof List) {
            for (Object table : (List<Object>) tables) {
                if (table instanceof Map) {
                    Map<String, Object> tableMap = (Map<String, Object>) table;
                    Object tableInfo = tableMap.get("info");
                    if (tableInfo instanceof JSONObject) {
                        String oldTalbeId =
                                StringBaseOpt.castObjectToString(((JSONObject) tableInfo).get("tableId"));
                        if (StringUtils.isNotBlank(oldTalbeId))
                            tableIds.add(oldTalbeId);
                    }
                }
            }
        }
        Object views = pm.getModuleDesign().get("views");
        if (views instanceof List) {
            for (Object view : (List<Object>) views) {
                if (view instanceof Map) {
                    Map<String, Object> viewMap = (Map<String, Object>) view;
                    Object viewInfo = viewMap.get("info");
                    if (viewInfo instanceof JSONObject) {
                        String oldTalbeId =
                                StringBaseOpt.castObjectToString(((JSONObject) viewInfo).get("viewId"));
                        if (StringUtils.isNotBlank(oldTalbeId))
                            tableIds.add(oldTalbeId);
                    }
                }
            }
        }

        if(tableIds.size()>0)
            return tableStructDao.listTablesByIds(tableIds);
        return null;
    }

    @Override
    public ProjectModule forkModule(CentitUserDetails userDetails, String moduleId, String projectId) {
        String userCode = userDetails.getUserCode();
        String userName = userDetails.getUserInfo().getString("userName");
        if(!projectTeamDao.checkMember(projectId, userCode)){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "用户 "+userCode+" 无权操作 " + projectId+" 项目。");
        }
        ProjectModule pm = projectModuleDao.getObjectById(moduleId);

        //更新排序评分
        ProjectInfo projectInfo = projectInfoDao.getObjectById(pm.getProjectId());
        projectInfo.setTableForkedTimes(projectInfo.getTableForkedTimes()+2);
        projectInfoDao.updateSortScore(projectInfo);


        pm.setModuleId(UuidOpt.getUuidAsString22());
        pm.setProjectId(projectId);
        pm.setCreatorCode(userCode);

        Object tables = pm.getModuleDesign().get("tables");
        if (tables instanceof List) {
            for (Object table : (List<Object>) tables) {
                if (table instanceof Map) {
                    Map<String, Object> tableMap = (Map<String, Object>) table;
                    Object tableInfo = tableMap.get("info");
                    if (tableInfo instanceof JSONObject) {
                        String oldTalbeId =
                                StringBaseOpt.castObjectToString(((JSONObject) tableInfo).get("tableId"));
                        String newTableId = UuidOpt.getUuidAsString22();
                        TableStruct ts = tableStructDao.getObjectById(oldTalbeId);
                        if(ts!=null) {
                            ts.setTableId(newTableId);
                            ts.setCreatorCode(userCode);
                            ts.setCreatorName(userName);
                            tableStructDao.mergeTable(ts, projectId);
                        }
                        ((JSONObject) tableInfo).put("tableId", newTableId);
                    }
                }
            }
        }

        Object views = pm.getModuleDesign().get("views");
        if (views instanceof List) {
            for (Object view : (List<Object>) views) {
                if (view instanceof Map) {
                    Map<String, Object> viewMap = (Map<String, Object>) view;
                    Object viewInfo = viewMap.get("info");
                    if (viewInfo instanceof JSONObject) {
                        String oldTalbeId =
                                StringBaseOpt.castObjectToString(((JSONObject) viewInfo).get("viewId"));
                        String newTableId = UuidOpt.getUuidAsString22();
                        TableStruct ts = tableStructDao.getObjectById(oldTalbeId);
                        if(ts!=null) {
                            ts.setTableId(newTableId);
                            ts.setCreatorCode(userCode);
                            ts.setCreatorName(userName);
                            tableStructDao.mergeTable(ts, projectId);
                        }
                        ((JSONObject) viewInfo).put("viewId", newTableId);
                    }
                }
            }
        }

        projectModuleDao.saveNewObject(pm);
        return pm;
    }

    @Override
    public String makeCreateSql(String moduleId, DBType dbType) {
        //TODO 需要添加关联关系的sql语句
        ProjectModule module = projectModuleDao.getObjectById(moduleId);
        List<String> tables = module.extraReferenceTables("VT");
        StringBuilder sqlBuilder = new StringBuilder("/*==========================================*/\r\n" +
                "/* Created on:   ").append(DatetimeOpt.currentDatetime()).append("                        */\r\n" +
                "/*==========================================*/\r\n");

        StringBuilder viewSqlBuilder = new StringBuilder();
        if(tables!=null){
            for(String ts : tables){
                TableStruct struct = tableStructDao.getObjectById(ts);
                if(struct!=null) {
                    if("V".equals(struct.getTableType())) {
                        viewSqlBuilder.append("/*===  ").append(struct.getTableLabelName()).append("  =======*/\r\n")
                                .append(struct.buildCreateTableSql(dbType)).append("\r\n\r\n");
                    } else {
                        sqlBuilder.append("/*===  ").append(struct.getTableLabelName()).append("  =======*/\r\n")
                                .append(struct.buildCreateTableSql(dbType)).append("\r\n\r\n");
                    }
                }
            }
        }
        sqlBuilder.append(viewSqlBuilder);
        return sqlBuilder.toString();
    }


    @Override
    public JSONObject exportModule(String moduleId) {

        ProjectModule module = projectModuleDao.getObjectById(moduleId);
        if(module==null){
            return null;
        }

        JSONObject project = new JSONObject();
        project.put("projectInfo", projectInfoDao.getObjectById(module.getProjectId()));

        JSONArray jaModules = new JSONArray();
        jaModules.add(module.toModuleJson());
        project.put("modules", jaModules);


        List<String> tables = module.extraReferenceTables("VT");
        if(tables!=null && tables.size()>0) {
            JSONArray jaTables = new JSONArray();
            for(String ts : tables){
                TableStruct table = tableStructDao.getObjectById(ts);
                if(table== null) continue;
                jaTables.add(table.toTableJson());
            }
            project.put("tables", jaTables);
        }
        return project;
    }

}
