package com.centit.tablestore.po;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.alibaba.fastjson2.annotation.JSONField;
import com.centit.framework.core.dao.DictionaryMap;
import com.centit.support.algorithm.BooleanBaseOpt;
import com.centit.support.algorithm.NumberBaseOpt;
import com.centit.support.algorithm.StringBaseOpt;
import com.centit.support.database.ddl.GeneralDDLOperations;
import com.centit.support.database.metadata.SimpleTableField;
import com.centit.support.database.metadata.SimpleTableInfo;
import com.centit.support.database.orm.GeneratorCondition;
import com.centit.support.database.orm.GeneratorType;
import com.centit.support.database.orm.ValueGenerator;
import com.centit.support.database.utils.DBType;
import com.centit.support.database.utils.FieldType;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.persistence.*;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.Pattern;
import java.sql.SQLException;
import java.util.*;

@ApiModel
@Entity
@Table(name = "TS_TABLE_STRUCT")
public class TableStruct implements java.io.Serializable {
    private static final long serialVersionUID = 1L;

    @Getter @Setter @Id
    @Column(name = "TABLE_ID")
    @ApiModelProperty(value = "表ID")
    @ValueGenerator(strategy = GeneratorType.UUID22)
    private String tableId;

    @Getter @Setter
    @Column(name = "PROJECT_ID")
    @ApiModelProperty(value = "项目ID")
    private String projectId;

    /**
     * 类别 表 T table /视图 V view / C-大字段 目前只支持json格式
     */
    @Getter @Setter
    @Column(name = "TABLE_TYPE")
    @NotBlank(message = "字段不能为空")
    @Pattern(regexp = "[TVC]")
    @Length(max = 1, message = "字段长度不能大于{max}")
    @ApiModelProperty(value = "表类别（T-表；V-视图；C-大字段）")
    @DictionaryMap(fieldName = "tableTypeText", value = "TableType")
    private String tableType;

    /**
     * 表代码/表名
     */
    @Getter @Setter
    @Column(name = "TABLE_NAME")
    @NotBlank(message = "字段不能为空")
    @Length(max = 64, message = "字段长度不能大于{max}")
    @ApiModelProperty(value = "表名")
    @OrderBy
    private String tableName;

    /**
     * 表中文名称
     */
    @Getter @Setter
    @Column(name = "TABLE_LABEL_NAME")
    @NotBlank(message = "字段不能为空")
    @Length(max = 100, message = "字段长度不能大于{max}")
    @ApiModelProperty(value = "表中文名")
    private String tableLabelName;

    /**
     * 描述
     */
    @Getter @Setter
    @ApiModelProperty(value = "表描述")
    @Column(name = "TABLE_COMMENT")
    @Length(max = 256, message = "字段长度不能大于{max}")
    private String tableComment;

    @Getter @Setter
    @Column(name = "CREATOR_CODE")
    @ApiModelProperty(value = "表创建人")
    private String creatorCode;

    @Getter @Setter
    @Column(name = "CREATOR_NAME")
    @ApiModelProperty(value = "表创建人姓名")
    private String creatorName;

    @Getter @Setter
    @Column(name = "LAST_UPDATE_TIME")
    @ApiModelProperty(value = "最后更新时间")
    @ValueGenerator(strategy = GeneratorType.FUNCTION, condition = GeneratorCondition.ALWAYS, value = "today()")
    private Date lastUpdateTIme;

    @Getter @Setter
    @Basic(fetch = FetchType.LAZY)
    @Column(name = "COLUMN_FIELD_JSON")
    @ApiModelProperty(value = "字段信息JSON")
    private JSONObject metadataJson;

    @Column(name = "FULLTEXT_INDEX")
    @Basic(fetch = FetchType.LAZY)
    @JSONField(serialize = false, deserialize = false)
    @ApiModelProperty(value = "全文检索字段")
    private String fulltextIndex;

    public TableStruct(){
        this.metadataJson = new JSONObject();
        this.tableType = "T";
    }

    public String getFulltextIndex(){
        StringBuilder textBuilder =
                new StringBuilder(tableName).append(" ").append(tableLabelName).append(" ").append(tableComment).append(";");
        Object columns = metadataJson.get("columns");
        if(columns instanceof List){
            for(Object col : (List<Object>) columns){
                if(col instanceof Map){
                    Map<String, Object> colMap = (Map<String, Object>)col;
                    textBuilder.append(" ").append(colMap.get("columnName"))
                            .append(" ").append(colMap.get("fieldLabelName"));
                    Object comments = colMap.get("columnComment");
                    if(comments !=null) {
                        textBuilder.append(" ").append(comments).append(";");
                    }
                }
            }
        }
        fulltextIndex = textBuilder.toString();
        return fulltextIndex;
    }

    public List<TableIndexInfo> fetchIndexes(){
        if(metadataJson==null)
            return null;
        JSONArray indexes = metadataJson.getJSONArray("indexes");
        if(indexes==null || indexes.size()==0)
            return null;

        List<TableIndexInfo> tableIndexes = new ArrayList<>();
        for(Object obj : indexes){
            if(obj instanceof JSONObject){
                JSONObject indJson = (JSONObject) obj;
                TableIndexInfo ind = indJson.toJavaObject(TableIndexInfo.class);
                tableIndexes.add(ind);
            }
        }
        return tableIndexes;
    }

    @Getter @Setter
    @Column(name = "VIEW_TIMES")
    @ApiModelProperty(value = "被查看次数，复制sql语句、导出视为有效查看")
    private Long viewTimes;

    public String extraViewSql(){
        return StringBaseOpt.castObjectToString(metadataJson.get("viewSql"));
    }

    public SimpleTableInfo toTableInfo(DBType dt){
        SimpleTableInfo tableInfo = new SimpleTableInfo();
        tableInfo.setTableName(this.tableName);
        tableInfo.setTableLabelName(this.tableLabelName);
        tableInfo.setTableComment(this.tableComment);
        tableInfo.setTableType(this.tableType);
        if(metadataJson!=null) {
            Object columns = metadataJson.get("columns");
            if (columns instanceof List) {
                for (Object col : (List<Object>) columns) {
                    if (col instanceof Map) {
                        SimpleTableField column = new SimpleTableField();
                        Map<String, Object> colMap = (Map<String, Object>) col;
                        column.setColumnName(StringBaseOpt.castObjectToString(colMap.get("columnName")));
                        column.setFieldLabelName(StringBaseOpt.castObjectToString(colMap.get("fieldLabelName")));
                        column.setColumnComment(StringBaseOpt.castObjectToString(colMap.get("columnComment")));
                        //column.setColumnType(StringBaseOpt.castObjectToString(colMap.get("columnType")));
                        column.setMandatory(BooleanBaseOpt.castObjectToBoolean(colMap.get("mandatory"), false));
                        column.setFieldType(StringBaseOpt.castObjectToString(colMap.get("fieldType")));
                        column.setColumnType(FieldType.mapToDatabaseType(column.getFieldType(), dt));
                        column.setMaxLength(NumberBaseOpt.castObjectToInteger(colMap.get("maxLength"), 0));
                        column.setPrimaryKey(BooleanBaseOpt.castObjectToBoolean(colMap.get("primaryKey"), false));
                        //公用属性
                        column.setScale(NumberBaseOpt.castObjectToInteger(colMap.get("scale"), 0));
                        column.setPrecision(NumberBaseOpt.castObjectToInteger(colMap.get("precision"), 0));
                        if(StringUtils.isNotBlank(column.getColumnName())) {
                            tableInfo.addColumn(column);
                        }
                    }
                }
            }
        }
        return tableInfo;
    }

    @JSONField(deserialize = false)
    @ApiModelProperty(hidden = true)
    public void setTableColumns(Collection<SimpleTableField> columns){
        if(metadataJson==null){
            metadataJson = new JSONObject();
        }
        metadataJson.put("columns", JSON.toJSON(columns));
    }

    public JSONObject toTableJson(){
        JSONObject jt = this.getMetadataJson();
        if (jt == null) {
            jt = new JSONObject();
        }
        JSONObject tableInfo = new JSONObject();
        tableInfo.put("tableId", this.getTableId());
        tableInfo.put("tableType", this.getTableType());
        tableInfo.put("tableName", this.getTableName());
        tableInfo.put("tableLabelName", this.getTableLabelName());
        tableInfo.put("tableComment", this.getTableComment());
        jt.put("tableInfo", tableInfo);
        return jt;
    }

    public static TableStruct formTableJson(JSONObject jt){
        TableStruct ts = new TableStruct();
        JSONObject tableInfo = jt.getJSONObject("tableInfo");
        if(tableInfo!=null){
            ts.setTableId(tableInfo.getString("tableId"));
            ts.setTableType(tableInfo.getString("tableType"));
            ts.setTableName(tableInfo.getString("tableName"));
            ts.setTableLabelName(tableInfo.getString("tableLabelName"));
            ts.setTableComment(tableInfo.getString("tableComment"));
        }

        JSONObject metadataJson = new JSONObject();
        metadataJson.put("columns", jt.get("columns"));
        metadataJson.put("indexes", jt.get("indexes"));
        metadataJson.put("viewSql", jt.get("viewSql"));

        ts.setMetadataJson(metadataJson);
        return ts;
    }


    public String buildCreateTableSql(final DBType dbType){
        // 完善sql语句的生成， 这个create view 需要迁移到 DDL 中
        // 需要添加 索引的语句生成表达式
        try {
            StringBuilder sqlBuilder = new StringBuilder();
            if ("V".equals(this.getTableType())) {
                sqlBuilder.append("create view ").append(this.getTableName()).append(" as \r\n  ")
                        .append(this.extraViewSql()).append(";");
            } else {
                sqlBuilder.append(GeneralDDLOperations.createDDLOperations(dbType)
                        .makeCreateTableSql(this.toTableInfo(dbType), true)).append(";");
                // TODO 添加 索引的语句生成表达式
                List<TableIndexInfo> indexes = this.fetchIndexes();
                if(indexes!=null && indexes.size()>0){
                    for(TableIndexInfo ind : indexes){
                        sqlBuilder.append("\r\n").append(
                                ind.buildCreateIndexSql(this.getTableName(), dbType)).append(";");
                    }
                }
            }

            return sqlBuilder.toString();
        } catch (SQLException sqlExp){
            return "/** 创建表：" + this.getTableLabelName()+"("+this.getTableName() +
                    ")时出错，错误信息为：" + sqlExp.getMessage() +"*/";
        }
    }
}
