package com.centit.tablestore.controller;

import com.alibaba.fastjson2.JSONObject;
import com.centit.fileserver.utils.UploadDownloadUtils;
import com.centit.framework.core.controller.BaseController;
import com.centit.framework.core.controller.WrapUpResponseBody;
import com.centit.framework.core.dao.PageQueryResult;
import com.centit.support.algorithm.BooleanBaseOpt;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.PageDesc;
import com.centit.tablestore.po.ProjectHistory;
import com.centit.tablestore.service.ProjectHistoryService;
import com.centit.tablestore.service.ProjectInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;

@Api(value = "历史版本管理", tags = "历史版本管理")
@RestController
@RequestMapping(value = "/history")
public class ProjectHistoryController extends BaseController {

    @Autowired
    private ProjectHistoryService projectHistoryService;

    @Autowired
    protected ProjectInfoService projectInfoService;

    @ApiOperation(value = "新增版本信息; 存储当前版本！")
    @ApiImplicitParam(name = "includeContent", type = "query", value = "是否已包含内容信息")
    @PostMapping()
    @WrapUpResponseBody
    public String createHistoryVersion(@RequestBody ProjectHistory historyVersion, String includeContent){
        if(! BooleanBaseOpt.castObjectToBoolean(includeContent, false)) {
            JSONObject jsonObject = projectInfoService.exportProject(historyVersion.getProjectId());
            historyVersion.setStructContent(jsonObject);
        }
        projectHistoryService.saveProjectHistory(historyVersion);
        return historyVersion.getHistoryId();
    }

    @ApiOperation(value = "修改版本信息")
    @PutMapping()
    @WrapUpResponseBody
    public void updateHistory(@RequestBody ProjectHistory historyVersion) {
        projectHistoryService.updateProjectHistory(historyVersion);
    }

    @ApiOperation(value = "删除版本信息")
    @DeleteMapping(value = "/{historyId}")
    @WrapUpResponseBody
    public void deleteHistory(@PathVariable String historyId) {
        projectHistoryService.deleteProjectHistory(historyId);
    }

    @ApiOperation(value = "查询版本信息列表")
    @GetMapping("/list")
    @WrapUpResponseBody
    public PageQueryResult<ProjectHistory> listHistory(HttpServletRequest request, PageDesc pageDesc) {
        List<ProjectHistory> list = projectHistoryService.listProjectHistory(BaseController.collectRequestParameters(request), pageDesc);
        return PageQueryResult.createResult(list, pageDesc);
    }

    @ApiOperation(value = "查询单个版本信息")
    @GetMapping(value = "/{historyId}")
    @WrapUpResponseBody
    public ProjectHistory getHistory(@PathVariable String historyId) {
        return projectHistoryService.getProjectHistory(historyId);
    }

    @ApiOperation(value = "回退到某个具体的版本")
    @PostMapping("/restore")
    @WrapUpResponseBody
    public void restoreHistoryVersion(@RequestBody ProjectHistory historyVersion){
        projectHistoryService.restoreProjectHistory(historyVersion.getHistoryId());
    }

    @ApiOperation(value = "导出历史表结构信息", notes = "导出历史表结构信息")
    @ApiImplicitParam(name = "historyId", type = "path", value = "历史版本ID")
    @GetMapping("/export/{historyId}")
    public void exportProject(@PathVariable String historyId, HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        ProjectHistory historyInfo = projectHistoryService.getProjectHistory(historyId);
        if(historyInfo==null){
            throw new ObjectException(ObjectException.DATA_NOT_FOUND_EXCEPTION ,"历史版本："+historyId+" 不存在！");
        }
        JSONObject json = historyInfo.getStructContent();
        String fileName = historyInfo.getHistoryLabel()+".json";
        ByteArrayInputStream bis = new ByteArrayInputStream(json.toJSONString().getBytes(StandardCharsets.UTF_8));
        UploadDownloadUtils.downloadFile(bis, fileName, request, response);
    }
}
