package com.centit.tablestore.controller;

import com.alibaba.fastjson2.JSONObject;
import com.centit.fileserver.utils.UploadDownloadUtils;
import com.centit.framework.common.ResponseData;
import com.centit.framework.common.WebOptUtils;
import com.centit.framework.core.controller.BaseController;
import com.centit.framework.core.controller.WrapUpContentType;
import com.centit.framework.core.controller.WrapUpResponseBody;
import com.centit.framework.security.model.CentitUserDetails;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.DBType;
import com.centit.tablestore.po.ProjectModule;
import com.centit.tablestore.po.TableStruct;
import com.centit.tablestore.service.ProjectModuleService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;

@Api(value = "模块维护接口", tags = "模块维护接口")
@RestController
@RequestMapping(value = "module")
public class ModuleController extends BaseController {

    @Autowired
    protected ProjectModuleService projectModuleService;

    @ApiOperation(value = "获取项目所有模块", notes = "获取项目所有模块不分页")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping("/list/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public List<ProjectModule> list(@PathVariable String projectId) {
        return projectModuleService.listProjectModules(projectId);
    }

    @ApiOperation(value = "查询单个模块")
    @ApiImplicitParam(name = "moduleId", type = "path", value = "模块ID")
    @GetMapping(value = "/{moduleId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public ProjectModule getProjectModule(@PathVariable String moduleId) {
        return projectModuleService.getProjectModule(moduleId);
    }

    @ApiOperation(value = "查询模块关联的表和视图")
    @ApiImplicitParam(name = "moduleId", type = "path", value = "模块ID")
    @GetMapping(value = "/tables/{moduleId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public List<TableStruct> listModuleTables(@PathVariable String moduleId) {
        return projectModuleService.listModuleTables(moduleId);
    }

    @ApiOperation(value = "删除单个模块", notes = "删除单个模块，不会删除表")
    @ApiImplicitParam(name = "moduleId", type = "path", value = "模块ID")
    @DeleteMapping(value = "/{moduleId}")
    @WrapUpResponseBody
    public void deleteProjectModule(@PathVariable String moduleId) {
        projectModuleService.deleteProjectModule(moduleId);
    }

    @ApiOperation(value = "新建模块数据")
    @PostMapping
    @WrapUpResponseBody
    public ProjectModule saveProjectModule(@RequestBody ProjectModule projectModule){
        projectModuleService.saveProjectModule(projectModule);
        return projectModule;
    }

    @ApiOperation(value = "修改模块数据")
    @PutMapping
    @WrapUpResponseBody
    public void updateProjectModule(@RequestBody ProjectModule projectModule){
        projectModuleService.updateProjectModule(projectModule);
    }

    @ApiOperation(value = "fork模块数据")
    @ApiImplicitParams({
    @ApiImplicitParam(name = "moduleId", type = "path", value = "模块ID"),
    @ApiImplicitParam(name = "projectId", type = "path", value = "目标项目ID，必须是当前用户可以操作的项目")})
    @PostMapping("/fork/{moduleId}/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public ProjectModule forkModuleInfo(@PathVariable String moduleId, @PathVariable String projectId, HttpServletRequest request){
        CentitUserDetails ud = WebOptUtils.getCurrentUserDetails(request);
        if(ud == null){
            throw new ObjectException(ResponseData.ERROR_FORBIDDEN, "Session过期或者用户没有登录。");
        }
        return projectModuleService.forkModule(ud, moduleId, projectId);
    }

    @ApiOperation(value = "生成脚本", notes = "生成脚本")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "moduleId", type = "path", value = "模块ID"),
            @ApiImplicitParam(name = "dbtype", type = "query", value = "数据类别")})
    @GetMapping("/sql/{moduleId}")
    @WrapUpResponseBody
    public String createSql(@PathVariable String moduleId, String dbtype) {
        return projectModuleService.makeCreateSql(moduleId, DBType.mapDBType(dbtype, DBType.MySql));
    }

    @ApiOperation(value = "DDL脚本以文件方式返回", notes = "DDL脚本以文件方式返回")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "moduleId", type = "path", value = "模块ID"),
            @ApiImplicitParam(name = "dbtype", type = "query", value = "数据类别")})
    @GetMapping("/ddl/{moduleId}")
    public void downloadDDL(@PathVariable String moduleId, String dbtype, HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        ProjectModule module = projectModuleService.getProjectModule(moduleId);
        String ddl = projectModuleService.makeCreateSql(moduleId, DBType.mapDBType(dbtype, DBType.MySql));
        String fileName = module.getModuleName()+".sql";
        ByteArrayInputStream bis = new ByteArrayInputStream(ddl.getBytes(StandardCharsets.UTF_8));
        UploadDownloadUtils.downloadFile(bis, fileName, request, response);
    }

    @ApiOperation(value = "导出项目的表结构信息", notes = "导出项目的表结构信息")
    @ApiImplicitParam(name = "moduleId", type = "path", value = "模块ID")
    @GetMapping("/export/{moduleId}")
    public void exportModule(@PathVariable String moduleId, HttpServletRequest request, HttpServletResponse response)
            throws IOException {
        ProjectModule module = projectModuleService.getProjectModule(moduleId);
        if(module==null){
            throw new ObjectException(ObjectException.DATA_NOT_FOUND_EXCEPTION ,"模块："+moduleId+" 不存在！");
        }
        JSONObject json = projectModuleService.exportModule(moduleId);
        String fileName = module.getModuleName()+".json";
        ByteArrayInputStream bis = new ByteArrayInputStream(json.toJSONString().getBytes(StandardCharsets.UTF_8));
        UploadDownloadUtils.downloadFile(bis, fileName, request, response);
    }

}
