package com.centit.tablestore.controller;

import com.alibaba.fastjson2.JSONArray;
import com.centit.framework.common.WebOptUtils;
import com.centit.framework.core.controller.BaseController;
import com.centit.framework.core.controller.WrapUpContentType;
import com.centit.framework.core.controller.WrapUpResponseBody;
import com.centit.framework.core.dao.DictionaryMapUtils;
import com.centit.framework.core.dao.PageQueryResult;
import com.centit.support.database.utils.DBType;
import com.centit.support.database.utils.PageDesc;
import com.centit.tablestore.po.ProjectInfo;
import com.centit.tablestore.po.TableStruct;
import com.centit.tablestore.service.TableStructService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Map;

@Api(value = "表结构维护接口", tags = "表结构维护接口")
@RestController
@RequestMapping(value = "table")
public class TableController extends BaseController {

    @Autowired
    protected TableStructService tableStructService;

    @ApiOperation(value = "获取项目所有表", notes = "获取项目所有表")
    @ApiImplicitParams({
        @ApiImplicitParam(
                name = "pageDesc", value = "json格式，分页对象信息",
                paramType = "body", dataTypeClass = PageDesc.class),
        @ApiImplicitParam(name = "projectId", type = "path", value = "表结构ID")})
    @GetMapping("/list/{projectId}")
    @WrapUpResponseBody
    public PageQueryResult<TableStruct> list(@PathVariable String projectId, PageDesc pageDesc, HttpServletRequest request) {
        Map<String, Object> filterMap = collectRequestParameters(request);
        List<TableStruct> objs = tableStructService.listProjectTables(projectId, filterMap, pageDesc);
        return PageQueryResult.createResult(objs, pageDesc);
    }

    @ApiOperation(value = "查询单个表")
    @ApiImplicitParam(name = "tableId", type = "path", value = "表结构ID")
    @GetMapping(value = "/{tableId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public TableStruct getTableStruct(@PathVariable String tableId) {
        return tableStructService.getTableStruct(tableId);
    }

    @ApiOperation(value = "删除单个表结构", notes = "删除单个表结构，不会删除表")
    @ApiImplicitParam(name = "tableId", type = "path", value = "表结构ID")
    @DeleteMapping(value = "/{tableId}")
    @WrapUpResponseBody
    public void deleteTableStruct(@PathVariable String tableId) {
        tableStructService.deleteTableStruct(tableId);
    }

    @ApiOperation(value = "新建表结构数据")
    @PostMapping
    @WrapUpResponseBody
    public void saveTableStruct(@RequestBody TableStruct TableStruct){
        tableStructService.saveTableStruct(TableStruct);
    }

    @ApiOperation(value = "新建表结构数据")
    @PostMapping("/updateViewSql")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public TableStruct updateViewSql(@RequestBody TableStruct tableStruct){
        // 分析sql语句，更新 试图对应的字段信息
        return tableStructService.updateViewSql(tableStruct);
    }

    @ApiOperation(value = "修改表结构数据")
    @PutMapping
    @WrapUpResponseBody
    public void updateTableStruct(@RequestBody TableStruct TableStruct){
        tableStructService.updateTableStruct(TableStruct);
    }
    @ApiOperation(value = "fork表结构数据")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tableId", type = "path", value = "表结构ID"),
            @ApiImplicitParam(name = "projectId", type = "path", value = "目标项目ID，必须是当前用户可以操作的项目")})
    @PostMapping("/fork/{tableId}/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public TableStruct forkTableInfo(@PathVariable String tableId, @PathVariable String projectId, HttpServletRequest request){
        String userCode = WebOptUtils.getCurrentUserCode(request);
        return tableStructService.forkTable(userCode, tableId, projectId);
    }

    @ApiOperation(value = "生成创建表结构的脚本", notes = "生成创建表结构的脚本")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "tableId", type = "path", value = "表结构ID"),
            @ApiImplicitParam(name = "dbtype", type = "query", value = "数据类别")})
    @GetMapping("/sql/{tableId}")
    @WrapUpResponseBody
    public String createSql(@PathVariable String tableId, String dbtype) {
        return tableStructService.makeCreateSql(tableId, DBType.mapDBType(dbtype));
    }

    @ApiOperation(value = "全文检索表", notes = "全文检索表、用于首页的检索")
    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "pageDesc", value = "json格式，分页对象信息",
                    paramType = "body", dataTypeClass = PageDesc.class),
            @ApiImplicitParam(name = "keywords", type = "query", value = "表结构ID"),
            @ApiImplicitParam(name = "catalog", type = "query", value = "表结构ID")})
    @GetMapping("/search")
    @WrapUpResponseBody
    public PageQueryResult<Object> search(String keywords, String catalog, PageDesc pageDesc) {
        JSONArray objs = tableStructService.searchTables(keywords, catalog, pageDesc);
        return PageQueryResult.createJSONArrayResult(objs, pageDesc, TableStruct.class);
    }

    @ApiOperation(value = "检索表统计信息", notes = "全文检索表只返回统计信息，用于搜索结果分类")
    @ApiImplicitParam(name = "keywords", type = "query", value = "表结构ID")
    @GetMapping("/stat")
    @WrapUpResponseBody
    public JSONArray stat(String keywords) {
        JSONArray objs = tableStructService.statTables(keywords);
        return DictionaryMapUtils.mapJsonArray(objs, ProjectInfo.class);
    }
}
