package com.centit.tablestore.controller;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.centit.framework.common.WebOptUtils;
import com.centit.framework.core.controller.BaseController;
import com.centit.framework.core.controller.WrapUpContentType;
import com.centit.framework.core.controller.WrapUpResponseBody;
import com.centit.framework.core.dao.PageQueryResult;
import com.centit.support.algorithm.BooleanBaseOpt;
import com.centit.support.algorithm.StringBaseOpt;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.DBType;
import com.centit.support.database.utils.PageDesc;
import com.centit.tablestore.po.ProjectInfo;
import com.centit.tablestore.po.ProjectTeam;
import com.centit.tablestore.service.ProjectInfoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.List;

@Api(value = "项目维护接口", tags = "项目维护接口")
@RestController
@RequestMapping(value = "project")
public class ProjectController extends BaseController {

    @Autowired
    protected ProjectInfoService projectInfoService;

    @ApiOperation(value = "获取用户所有项目", notes = "获取用户所有项目")
    @ApiImplicitParams({
    @ApiImplicitParam(
            name = "pageDesc", value = "json格式，分页对象信息",
            paramType = "body", dataTypeClass = PageDesc.class),
    @ApiImplicitParam(
            name = "onlyCreated", value = "仅仅返回用户创建的项目",
            paramType = "query"),
    @ApiImplicitParam(
            name = "userCode", value = "用户代码，优先获取当前用户代码",
            paramType = "query")})
    @GetMapping
    @WrapUpResponseBody
    public PageQueryResult<Object> list(PageDesc pageDesc, String onlyCreated, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        //Map<String, Object> filterMap = collectRequestParameters(request);
        if(StringUtils.isBlank(userCode)){
            userCode = request.getParameter("userCode");
        }
        JSONArray projectInfos;
        if(BooleanBaseOpt.castObjectToBoolean(onlyCreated, false)) {
            projectInfos = projectInfoService.listUserCreateProjects(userCode, pageDesc);
        } else {
            projectInfos = projectInfoService.listUserProjects(userCode, pageDesc);
        }
        return PageQueryResult.createJSONArrayResult(projectInfos, pageDesc, ProjectInfo.class);
    }

    @ApiOperation(value = "查询单个项目")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping(value = "/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public ProjectInfo getProjectInfo(@PathVariable String projectId) {
        return projectInfoService.getProjectInfo(projectId);
    }

    @ApiOperation(value = "删除单个项目")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @DeleteMapping(value = "/{projectId}")
    @WrapUpResponseBody
    public void deleteProjectInfo(@PathVariable String projectId) {
        projectInfoService.deleteProjectInfo(projectId);
    }

    @ApiOperation(value = "新建项目数据")
    @PostMapping
    @WrapUpResponseBody
    public void saveProjectInfo(@RequestBody ProjectInfo projectInfo){
        projectInfoService.saveProjectInfo(projectInfo);
    }

    @ApiOperation(value = "修改项目数据")
    @PutMapping
    @WrapUpResponseBody
    public void updateProjectInfo(@RequestBody ProjectInfo projectInfo){
        projectInfoService.updateProjectInfo(projectInfo);
    }

    @ApiOperation(value = "生成脚本", notes = "生成脚本")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID"),
        @ApiImplicitParam(name = "dbtype", type = "query", value = "数据类别")})
    @GetMapping("/sql/{projectId}")
    @WrapUpResponseBody
    public String createSql(@PathVariable String projectId, String dbtype) {
        return projectInfoService.makeCreateSql(projectId, DBType.mapDBType(dbtype));
    }

    @ApiOperation(value = "fork项目数据")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @PostMapping("/fork/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public ProjectInfo forkProjectInfo(@PathVariable String projectId, HttpServletRequest request){
        String userCode = WebOptUtils.getCurrentUserCode(request);
        return projectInfoService.forkProject(userCode, projectId);
    }

    @ApiOperation(value = "查询项目协作人员")
    @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID")
    @GetMapping(value = "/member/{projectId}")
    @WrapUpResponseBody(contentType = WrapUpContentType.MAP_DICT)
    public List<ProjectTeam>  listProjectMember(@PathVariable String projectId, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        return projectInfoService.listProjectMember(userCode, projectId);
    }

    @ApiOperation(value = "添加项目协作人员")
    @PostMapping(value = "/member")
    @WrapUpResponseBody
    public void addProjectMember(@RequestBody String projectMembers, HttpServletRequest request) {
        //projectId projectMember
        String userCode = WebOptUtils.getCurrentUserCode(request);
        JSONObject projMembers = JSONObject.parseObject(projectMembers);
        String projectId = projMembers.getString("projectId");
        Object members = projMembers.get("projectMember");
        if(members == null )
            members = projMembers.get("projectMembers");
        if(StringUtils.isBlank(userCode) || StringUtils.isBlank(projectId) || members==null){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "表单数据格式不正确，或者用户没有登录。");
        }
        List<String> memeberCodes = StringBaseOpt.objectToStringList(members);
        for(String mc : memeberCodes) {
            projectInfoService.addProjectMember(userCode, projectId,mc);
        }
    }

    @ApiOperation(value = "删除项目协作人员")
    @ApiImplicitParams({
        @ApiImplicitParam(name = "projectId", type = "path", value = "项目ID"),
        @ApiImplicitParam(name = "memberCode", type = "path", value = "协作人员代码")})
    @DeleteMapping(value = "/delete/{projectId}/{memberCode}")
    @WrapUpResponseBody
    public void deleteProjectMember(@PathVariable String projectId, @PathVariable String memberCode, HttpServletRequest request) {
        String userCode = WebOptUtils.getCurrentUserCode(request);
        projectInfoService.deleteProjectMember(userCode, projectId, memberCode);
    }


    @ApiOperation(value = "删除项目协作人员,post方法，可以批量删除")
    @PostMapping(value = "/deleteMembers")
    @WrapUpResponseBody
    public void deleteProjectMember(@RequestBody String projectMembers, HttpServletRequest request) {
        //projectId projectMember
        String userCode = WebOptUtils.getCurrentUserCode(request);
        JSONObject projMembers = JSONObject.parseObject(projectMembers);
        String projectId = projMembers.getString("projectId");
        Object members = projMembers.get("projectMember");
        if(members == null )
            members = projMembers.get("projectMembers");
        if(StringUtils.isBlank(userCode) || StringUtils.isBlank(projectId) || members==null){
            throw new ObjectException(ObjectException.DATA_VALIDATE_ERROR, "表单数据格式不正确，或者用户没有登录。");
        }
        List<String> memeberCodes = StringBaseOpt.objectToStringList(members);
        for(String mc : memeberCodes) {
            projectInfoService.deleteProjectMember(userCode, projectId,mc);
        }
    }
}
