package com.centit.tablestore.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.centit.framework.jdbc.dao.DatabaseOptUtils;
import com.centit.support.algorithm.CollectionsOpt;
import com.centit.support.algorithm.DatetimeOpt;
import com.centit.support.algorithm.StringBaseOpt;
import com.centit.support.algorithm.UuidOpt;
import com.centit.support.common.ObjectException;
import com.centit.support.database.utils.DBType;
import com.centit.support.database.utils.PageDesc;
import com.centit.tablestore.dao.ProjectInfoDao;
import com.centit.tablestore.dao.ProjectModuleDao;
import com.centit.tablestore.dao.ProjectTeamDao;
import com.centit.tablestore.dao.TableStructDao;
import com.centit.tablestore.po.*;
import com.centit.tablestore.service.ProjectInfoService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service("projectInfoService")
@Transactional
public class ProjectInfoServiceImpl implements ProjectInfoService{

    @Autowired
    protected ProjectInfoDao projectInfoDao;

    @Autowired
    protected ProjectModuleDao projectModuleDao;

    @Autowired
    protected TableStructDao tableStructDao;

    @Autowired
    protected ProjectTeamDao projectTeamDao;

    @Override
    public void saveProjectInfo(ProjectInfo projectInfo) {
        projectInfoDao.saveNewObject(projectInfo);
        projectTeamDao.saveNewObject(
                new ProjectTeam(projectInfo.getProjectId(), projectInfo.getCreatorCode()));
    }

    @Override
    public void updateProjectInfo(ProjectInfo projectInfo) {
        projectInfoDao.updateObject(projectInfo);
    }

    @Override
    public void deleteProjectInfo(String projectId) {
        projectInfoDao.deleteObjectById(projectId);
        projectModuleDao.deleteProjectModule(projectId);
        tableStructDao.deleteProjectTable(projectId);
        projectTeamDao.deleteProjectTeam(projectId);
    }

    @Override
    public ProjectInfo getProjectInfo(String projectId) {
        return projectInfoDao.getObjectById(projectId);
    }

    @Override
    public JSONArray listUserProjects(String userCode, PageDesc pageDesc) {
        return DatabaseOptUtils.listObjectsBySqlAsJson( projectInfoDao,
        "select a.* from TS_PROJECT_INFO a join TS_PROJECT_TEAM b on(a.PROJECT_ID= b.PROJECT_ID) where b.PROJECT_MEMBER = ?",
                new Object[]{userCode},
                pageDesc);
    }

    @Override
    public JSONArray listUserCreateProjects(String userCode, PageDesc pageDesc) {
        return DatabaseOptUtils.listObjectsBySqlAsJson( projectInfoDao,
               "select a.* from TS_PROJECT_INFO a where a.CREATOR_CODE=?",
                new Object[]{userCode}, pageDesc);
    }

    @Override
    public void addProjectMember(String optUser, String projectId, String memberCode) {
        if(!projectInfoDao.checkOwner(projectId, optUser)){
            throw new ObjectException("用户 "+optUser+" 不是项目 " + projectId+" 的创建人，无权操作。");
        }
        ProjectTeam pt = projectTeamDao.getObjectById(new ProjectTeamId(projectId, memberCode));
        if(pt!=null)
            return;
        projectTeamDao.saveNewObject(
                new ProjectTeam(projectId, memberCode));
    }

    @Override
    public void deleteProjectMember(String optUser, String projectId, String memberCode) {
        if(!projectInfoDao.checkOwner(projectId, optUser)){
            throw new ObjectException("用户 "+optUser+" 不是项目 " + projectId+" 的创建人，无权操作。");
        }
        if(!optUser.equals(memberCode)){
            projectTeamDao.deleteObjectById(new ProjectTeamId(projectId, memberCode));
        }
    }

    @Override
    public List<ProjectTeam> listProjectMember(String optUser, String projectId) {
        if(!projectInfoDao.checkOwner(projectId, optUser)){
            throw new ObjectException("用户 "+optUser+" 不是项目 " + projectId+" 的创建人，无权操作。");
        }
        return projectTeamDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
    }

    @Override
    public ProjectInfo forkProject(String optUser, String projectId) {
        ProjectInfo projectInfo = projectInfoDao.getObjectById( projectId);
        if(projectInfo==null) {
            throw new ObjectException("项目 " + projectId+" 不存在。");
        }
        if(StringUtils.equals(optUser, projectInfo.getCreatorCode())) {
            throw new ObjectException("用户 " + optUser+" 不能fork自己的项目。");
        }

        ProjectInfo forkProject = projectInfoDao.getObjectByProperties(
                CollectionsOpt.createHashMap("creatorCode", optUser, "forkProject", projectId));

        if(forkProject!=null)
            return forkProject;
        //复制项目
        forkProject = projectInfo;
        forkProject.setProjectId(UuidOpt.getUuidAsString22());
        forkProject.setForkProject(projectId);
        forkProject.setCreatorCode(optUser);
        //colone 主表
        saveProjectInfo(forkProject);
        //colone 字表 TS_PROJECT_MODULE
        HashMap<String, String> tableIdMap = new HashMap<>(100);
        //colone 字表 TS_TABLE_STRUCT
        List<TableStruct> tables = tableStructDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
        if(tables!=null){
            for(TableStruct ts : tables){
                String newTableId = UuidOpt.getUuidAsString22();
                tableIdMap.put(ts.getTableId(), newTableId);
                ts.setTableId(newTableId);
                ts.setProjectId(forkProject.getProjectId());
                ts.setCreatorCode(optUser);
                tableStructDao.saveNewObject(ts);
            }
        }

        List<ProjectModule> modules = projectModuleDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
        if(modules!=null){
            for(ProjectModule pm : modules){
                pm.setModuleId(UuidOpt.getUuidAsString22());
                pm.setProjectId(forkProject.getProjectId());
                pm.setCreatorCode(optUser);

                Object moduleTables = pm.getModuleDesign().get("tables");
                if (moduleTables instanceof List) {
                    for (Object table : (List<Object>) moduleTables) {
                        if (table instanceof Map) {
                            Map<String, Object> tableMap = (Map<String, Object>) table;
                            Object tableInfo = tableMap.get("info");
                            if (tableInfo instanceof JSONObject) {
                                String oldTalbeId =
                                        StringBaseOpt.castObjectToString(((JSONObject) tableInfo).get("tableId"));
                                String newTableId = tableIdMap.get(oldTalbeId);
                                if(newTableId==null){
                                    newTableId = "";
                                }
                                ((JSONObject) tableInfo).put("tableId", newTableId);
                            }
                        }
                    }
                }

                Object views = pm.getModuleDesign().get("views");
                if (views instanceof List) {
                    for (Object view : (List<Object>) views) {
                        if (view instanceof Map) {
                            Map<String, Object> viewMap = (Map<String, Object>) view;
                            Object viewInfo = viewMap.get("info");
                            if (viewInfo instanceof JSONObject) {
                                String oldTalbeId =
                                        StringBaseOpt.castObjectToString(((JSONObject) viewInfo).get("viewId"));
                                String newTableId = tableIdMap.get(oldTalbeId);
                                if(newTableId==null){
                                    newTableId = "";
                                }
                                ((JSONObject) viewInfo).put("viewId", newTableId);
                            }
                        }
                    }
                }

                projectModuleDao.saveNewObject(pm);
            }
        }

        return forkProject;
    }

    @Override
    public String makeCreateSql(String projectId, DBType dbType) {
        StringBuilder sqlBuilder = new StringBuilder("/*==============================================================*/\n\n" +
                "/* Created on:   ").append(DatetimeOpt.currentDatetime()).append("                        */\n\n" +
                "/*==============================================================*/\r\n");
        List<TableStruct> tables = tableStructDao.listObjectsByProperties(CollectionsOpt.createHashMap("projectId", projectId));
        if(tables!=null){
            for(TableStruct ts : tables){
                sqlBuilder.append(ts.buildCreateTableSql(dbType)).append("\r\n");
            }
        }
        return sqlBuilder.toString();
    }
}
