package com.centit.tablestore.po;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.annotation.JSONField;
import com.centit.framework.core.dao.DictionaryMap;
import com.centit.support.algorithm.BooleanBaseOpt;
import com.centit.support.algorithm.NumberBaseOpt;
import com.centit.support.algorithm.StringBaseOpt;
import com.centit.support.database.ddl.GeneralDDLOperations;
import com.centit.support.database.metadata.SimpleTableField;
import com.centit.support.database.metadata.SimpleTableInfo;
import com.centit.support.database.metadata.TableInfo;
import com.centit.support.database.orm.GeneratorCondition;
import com.centit.support.database.orm.GeneratorType;
import com.centit.support.database.orm.ValueGenerator;
import com.centit.support.database.utils.DBType;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Getter;
import lombok.Setter;
import org.hibernate.validator.constraints.Length;

import javax.persistence.*;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.Pattern;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

@ApiModel
@Entity
@Table(name = "TS_TABLE_STRUCT")
public class TableStruct implements java.io.Serializable {
    private static final long serialVersionUID = 1L;

    @Getter @Setter @Id
    @Column(name = "TABLE_ID")
    @ApiModelProperty(value = "表ID")
    @ValueGenerator(strategy = GeneratorType.UUID22)
    private String tableId;

    @Getter @Setter
    @Column(name = "PROJECT_ID")
    @ApiModelProperty(value = "项目ID")
    private String projectId;

    /**
     * 类别 表 T table /视图 V view / C-大字段 目前只支持json格式
     */
    @Getter @Setter
    @Column(name = "TABLE_TYPE")
    @NotBlank(message = "字段不能为空")
    @Pattern(regexp = "[TVC]")
    @Length(max = 1, message = "字段长度不能大于{max}")
    @ApiModelProperty(value = "表类别（T-表；V-视图；C-大字段）")
    @DictionaryMap(fieldName = "tableTypeText", value = "TableType")
    private String tableType;

    /**
     * 表代码/表名
     */
    @Getter @Setter
    @Column(name = "TABLE_NAME")
    @NotBlank(message = "字段不能为空")
    @Length(max = 64, message = "字段长度不能大于{max}")
    @ApiModelProperty(value = "表名")
    private String tableName;

    /**
     * 表中文名称
     */
    @Getter @Setter
    @Column(name = "TABLE_LABEL_NAME")
    @NotBlank(message = "字段不能为空")
    @Length(max = 100, message = "字段长度不能大于{max}")
    @ApiModelProperty(value = "表中文名")
    private String tableLabelName;

    /**
     * 描述
     */
    @Getter @Setter
    @ApiModelProperty(value = "表描述")
    @Column(name = "TABLE_COMMENT")
    @Length(max = 256, message = "字段长度不能大于{max}")
    private String tableComment;

    @Getter @Setter
    @Column(name = "CREATOR_CODE")
    @ApiModelProperty(value = "项目创建人")
    @DictionaryMap(value="userCode", fieldName = "creatorName")
    private String creatorCode;

    @Getter @Setter
    @Column(name = "LAST_UPDATE_TIME")
    @ApiModelProperty(value = "最后更新时间")
    @ValueGenerator(strategy = GeneratorType.FUNCTION, condition = GeneratorCondition.ALWAYS, value = "today()")
    private Date lastUpdateTIme;

    @Getter @Setter
    @Basic(fetch = FetchType.LAZY)
    @Column(name = "COLUMN_FIELD_JSON")
    @ApiModelProperty(value = "字段信息JSON")
    private JSONObject metadataJson;

    @Column(name = "FULLTEXT_INDEX")
    @Basic(fetch = FetchType.LAZY)
    @JSONField(serialize = false, deserialize = false)
    @ApiModelProperty(value = "全文检索字段")
    private String fulltextIndex;


    public String getFulltextIndex(){
        StringBuilder textBuilder =
                new StringBuilder(tableName).append(" ").append(tableLabelName).append(" ").append(tableComment).append(";");
        Object columns = metadataJson.get("columns");
        if(columns instanceof List){
            for(Object col : (List<Object>) columns){
                if(col instanceof Map){
                    Map<String, Object> colMap = (Map<String, Object>)col;
                    textBuilder.append(" ").append(colMap.get("columnName"))
                            .append(" ").append(colMap.get("fieldLabelName"))
                            .append(" ").append(colMap.get("columnComment")).append(";");
                }
            }
        }
        fulltextIndex = textBuilder.toString();
        return fulltextIndex;
    }

    @Getter @Setter
    @Column(name = "VIEW_TIMES")
    @ApiModelProperty(value = "被查看次数，复制sql语句、导出视为有效查看")
    private String viewTimes;

    public String extraViewSql(){
        return StringBaseOpt.castObjectToString(metadataJson.get("viewSql"));
    }

    public SimpleTableInfo toTableInfo(){
        SimpleTableInfo tableInfo = new SimpleTableInfo();
        tableInfo.setTableName(this.tableName);
        tableInfo.setTableLabelName(this.tableLabelName);
        tableInfo.setTableComment(this.tableComment);
        tableInfo.setTableType(this.tableType);
        Object columns = metadataJson.get("columns");
        if(columns instanceof List){
            for(Object col : (List<Object>) columns){
                if(col instanceof Map){
                    SimpleTableField column = new SimpleTableField();
                    Map<String, Object> colMap = (Map<String, Object>)col;
                    column.setColumnName(StringBaseOpt.castObjectToString(colMap.get("columnName")));
                    column.setFieldLabelName(StringBaseOpt.castObjectToString(colMap.get("fieldLabelName")));
                    column.setColumnComment(StringBaseOpt.castObjectToString(colMap.get("columnComment")));
                    column.setColumnType(StringBaseOpt.castObjectToString(colMap.get("columnType")));
                    column.setMandatory(BooleanBaseOpt.castObjectToBoolean(colMap.get("columnType"), false));
                    column.setFieldType(StringBaseOpt.castObjectToString(colMap.get("fieldType")));
                    column.setMaxLength(NumberBaseOpt.castObjectToInteger(colMap.get("maxLength")));
                    column.setScale(NumberBaseOpt.castObjectToInteger(colMap.get("scale")));
                    column.setPrimaryKey(BooleanBaseOpt.castObjectToBoolean(colMap.get("primaryKey"), false));
                    //公用属性
                    column.setPrecision(NumberBaseOpt.castObjectToInteger(colMap.get("maxLength")));
                    tableInfo.addColumn(column);
                }
            }
        }
        return tableInfo;
    }

    public void setTableColumns(Collection<SimpleTableField> columns){
        metadataJson.put("columns", JSON.toJSON(columns));
    }

    public String buildCreateTableSql(final DBType dbType){
        //TODO 完善sql语句的生成， 这个create view 需要迁移到 DDL 中
        // 需要添加 索引的语句生成表达式
        try {
            if ("V".equals(this.getTableType())) {
                return "create View " + this.getTableName() + " " + this.extraViewSql() + ";";
            } else {
                return GeneralDDLOperations.createDDLOperations(dbType).makeCreateTableSql(this.toTableInfo());
            }
        } catch (SQLException sqlExp){
            return "/** 创建表：" + this.getTableLabelName()+"("+this.getTableName() +
                    ")时出错，错误信息为：" + sqlExp.getMessage() +"*/";
        }
    }
}
