var img = 'data:image/png;base64,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';
function loadBlock5(year,city) {
    $.getJSON("js/baseUrl.json", function (rlt) {
        var baseURL = rlt.baseUrl;
        var cy_URL = baseURL + 'service/sjcj/dp/loadData?dataCode=zrc'
        if(year != undefined){
            if(year!= "全年"){
                cy_URL += '&dataYear='+year
            }
        }
        if(city != undefined){
            cy_URL += '&cityCode='+city
        }
        $.ajax({
            url:cy_URL,    //请求的url地址
            async: true,
            type: "get",
            success: function (data) {
                if (data != null) {
                    var trafficWay = [{
                        name: '城乡融合型',
                        value: data.城乡融合型
                    }, {
                        name: '其他',
                        value: data.其他
                    }, {
                        name: '规划新建型',
                        value: data.规划新建型
                    }, {
                        name: '特色保护型',
                        value: data.特色保护型
                    }, {
                        name: '聚集提升型',
                        value: data.聚集提升型
                    }
                    ];
                    loadEcharts(trafficWay);
                }
            },
            error: function (rlt, msg) {
                console.log("获取失败，原因：" + msg);
            }
        });
    })
}

function loadEcharts(trafficWay) {
    var data = [];
    var color = ['#00ffff', '#00cfff', '#006ced', '#ffe000', '#ffa800']
    for (var i = 0; i < trafficWay.length; i++) {
        data.push({
            value: trafficWay[i].value,
            name: trafficWay[i].name,
            itemStyle: {
                normal: {
                    borderWidth: 15,
                    shadowBlur: 20,
                    borderColor: color[i],
                    shadowColor: color[i]
                }
            }
        }, {
            value: 2,
            name: '',
            itemStyle: {
                normal: {
                    label: {
                        show: false
                    },
                    labelLine: {
                        show: false
                    },
                    color: 'rgba(0, 0, 0, 0)',
                    borderColor: 'rgba(0, 0, 0, 0)',
                    borderWidth: 0
                }
            }
        });
    }

    var seriesOption = [{
        name: '',
        type: 'pie',
        clockWise: false,
        radius: [150, 150],
        itemStyle: {
            normal: {
                label: {
                    show: true,
                    position: 'outside',
                    color: '#fff',
                    fontSize: 23,
                    formatter: function (params) {
                        var percent = 0;
                        var total = 0;
                        for (var i = 0; i < trafficWay.length; i++) {
                            total += trafficWay[i].value;
                        }
                        percent = ((params.value / total) * 100).toFixed(0);
                        if (params.name !== '') {
                            return '' + params.name + '\n' + '\n' + params.value + '  ' + percent + '%';
                        } else {
                            return '';
                        }
                    },
                },
                labelLine: {
                    length: 80,
                    length2: 60,
                    borderWidth: 8,
                    show: true,
                    color: '#00ffff'
                }
            }
        },
        data: data
    }];
    var myChart = echarts.init(document.getElementById('block5'));
    option = {
        color: color,
        graphic: {
            elements: [{
                type: "image",
                z: 10,
                style: {
                    image: img,
                    width: 400,
                    height: 400
                },
                left: 'center',
                top: 'center',
                position: [20, 20]
            }, {
                type: "text",
                left: "center",
                top: "center",
                style: {
                    text: "五市村庄\n类型分布",
                    textAlign: "center",
                    fill: "#ffffff",
                    fontSize: 40,
                }
            }]
        },
        tooltip: {
            show: false
        },
        toolbox: {
            show: false
        },
        series: seriesOption
    };
    // 使用刚指定的配置项和数据显示图表。
    myChart.setOption(option);

    myChart.currentIndex = -1;

    setInterval(function () {
        var dataLen = option.series[0].data.length;
        // 取消之前高亮的图形
        myChart.dispatchAction({
            type: 'downplay',
            seriesIndex: 0,
            dataIndex: myChart.currentIndex
        });
        myChart.currentIndex = (myChart.currentIndex + 1) % dataLen;
        // 高亮当前图形
        myChart.dispatchAction({
            type: 'highlight',
            seriesIndex: 0,
            dataIndex: myChart.currentIndex
        });
    }, 1000);
    window.addEventListener("resize", function () {
        myChart.resize();
    });
}
