package com.centit.tablestore.po;

import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.centit.framework.core.dao.DictionaryMap;
import com.centit.support.algorithm.StringBaseOpt;
import com.centit.support.database.orm.GeneratorCondition;
import com.centit.support.database.orm.GeneratorType;
import com.centit.support.database.orm.ValueGenerator;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;

import javax.persistence.*;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 项目模块
 */
@ApiModel
@Data
@Entity
@Table(name = "TS_PROJECT_MODULE")
public class ProjectModule implements java.io.Serializable {
    private static final long serialVersionUID = 1L;

    @Id
    @Column(name = "MODULE_ID")
    @ApiModelProperty(value = "模块ID")
    @ValueGenerator(strategy = GeneratorType.UUID22)
    private String moduleId;

    @Column(name = "PROJECT_ID")
    @ApiModelProperty(value = "项目ID")
    private String projectId;

    @Column(name = "MODULE_NAME")
    @ApiModelProperty(value = "模块名称")
    private String moduleName;

    @Column(name = "MODULE_DESC")
    @ApiModelProperty(value = "模块描述")
    private String moduleDesc;

    @Basic(fetch = FetchType.LAZY)
    @Column(name = "MODULE_DESIGN_JSON")
    @ApiModelProperty(value = "字段信息JSON")
    private JSONObject moduleDesign;

    @Column(name = "CREATOR_CODE")
    @ApiModelProperty(value = "模块创建人")
    @DictionaryMap(value="userCode", fieldName = "creatorName")
    private String creatorCode;

    @Column(name = "LAST_UPDATE_TIME")
    @ApiModelProperty(value = "最后更新时间")
    @ValueGenerator(strategy = GeneratorType.FUNCTION, condition = GeneratorCondition.ALWAYS, value = "today()")
    private Date lastUpdateTIme;

    public List<String> extraReferenceTables(String tableType) {
        if(moduleDesign==null){
            return null;
        }
        List<String> tableIds = new ArrayList<>();
        if (tableType.indexOf('T') >= 0) {
            Object tables = moduleDesign.get("tables");
            if (tables instanceof List) {
                for (Object table : (List<Object>) tables) {
                    if (table instanceof Map) {
                        Map<String, Object> tableMap = (Map<String, Object>) table;
                        Object tableInfo = tableMap.get("info");
                        if (tableInfo instanceof Map) {
                            tableIds.add(
                                    StringBaseOpt.castObjectToString(((Map<String, Object>) tableInfo).get("tableId")));
                        }
                    }
                }
            }
        }

        if (tableType.indexOf('V') >= 0) {
            Object views = moduleDesign.get("views");
            if (views instanceof List) {
                for (Object view : (List<Object>) views) {
                    if (view instanceof Map) {
                        Map<String, Object> viewMap = (Map<String, Object>) view;
                        Object viewInfo = viewMap.get("info");
                        if (viewInfo instanceof Map) {
                            tableIds.add(
                                    StringBaseOpt.castObjectToString(((Map<String, Object>) viewInfo).get("viewId")));
                        }
                    }
                }
            }
        }
        return tableIds;
    }

    public JSONObject toModuleJson(){
        JSONObject jm = this.getModuleDesign();
        if (jm == null) {
            jm = new JSONObject();
        }
        JSONObject moduleInfo = new JSONObject();
        moduleInfo.put("moduleId", this.getModuleId());
        moduleInfo.put("moduleName", this.getModuleName());
        moduleInfo.put("moduleDesc", this.getModuleDesc());
        jm.put("moduleInfo", moduleInfo);
        return jm;
    }

    public static ProjectModule formModuleJson(JSONObject jm){
        ProjectModule module = new ProjectModule();
        JSONObject moduleInfo = jm.getJSONObject("moduleInfo");
        if(moduleInfo!=null){
            module.setModuleId(moduleInfo.getString("moduleId"));
            module.setModuleName(moduleInfo.getString("moduleName"));
            module.setModuleDesc(moduleInfo.getString("moduleDesc"));
        }

        JSONObject moduleDesign = new JSONObject();
        moduleDesign.put("tables", jm.get("tables"));
        moduleDesign.put("views", jm.get("views"));
        moduleDesign.put("relations", jm.get("relations"));

        module.setModuleDesign(moduleDesign);
        return module;
    }

    private static String mapTalbeId(Map<String, String> tablesIdMap, String oldTableId){
        if(StringUtils.isBlank(oldTableId))
            return oldTableId;
        String newTableId = tablesIdMap.get(oldTableId);
        return StringUtils.isBlank(newTableId)? oldTableId : newTableId;
    }

    public void replaceModuleTablesId(Map<String, String> tablesIdMap){
        if(moduleDesign==null) return;
        //tables[].info.tableId
        JSONArray tables = moduleDesign.getJSONArray("tables");
        if(tables!=null && tables.size()>0) {
            for(Object obj : tables){
                if(obj instanceof JSONObject) {
                    JSONObject info = ((JSONObject) obj).getJSONObject("info");
                    if(info != null){
                        info.put("tableId", mapTalbeId(tablesIdMap, info.getString("tableId")));
                    }
                }
            }
        }

        //views[].info.viewId
        JSONArray views = moduleDesign.getJSONArray("views");
        if(views!=null && views.size()>0) {
            for(Object obj : views){
                if(obj instanceof JSONObject) {
                    JSONObject info = ((JSONObject) obj).getJSONObject("info");
                    if(info != null){
                        info.put("viewId", mapTalbeId(tablesIdMap, info.getString("viewId")));
                    }
                }
            }
        }

        //relations[].info.parentTable/chileTable
        JSONArray relations = moduleDesign.getJSONArray("relations");
        if(relations!=null && relations.size()>0) {
            for(Object obj : relations){
                if(obj instanceof JSONObject) {
                    JSONObject info = ((JSONObject) obj).getJSONObject("info");
                    if(info != null){
                        info.put("parentTable", mapTalbeId(tablesIdMap, info.getString("parentTable")));
                        info.put("chileTable", mapTalbeId(tablesIdMap, info.getString("chileTable")));
                    }
                }
            }
        }
    }
}
